<?php
namespace boru\backblaze;

use \boru\backblaze\BackBlazeBase;
use \boru\backblaze\BackBlazeBaseException;
use \boru\backblaze\Bucket;
use \boru\backblaze\BucketFile;
use \boru\dhutils\dhGlobal;
use \boru\dhutils\filesys\File;

class Client extends BackBlazeBase {
    public $apiUrl = 'https://api.backblazeb2.com';
    public $apiVersion = "/b2api/v2";

    public $debugDoTrace = false;

    public $uploadNormalLimit = 3000000000; //3gb
    public $multiPartSize = 100000000; //100mb
    public $bucketCacheTime = 300; //seconds

    protected $client;

    protected $cacheFile;

    protected $keyId;
    protected $applicationKey;
    protected $accountInfo;
    protected $uploadTries = 0;
    protected $uploadDelay = 1; //seconds
    protected $uploadBackoff = 1.2; //+20%

    protected $_bucketCacheLastList = 0;
    protected $_bucketCache = [];

    protected $debug_name = "backblaze";
    protected $debug_prefix = "BACKBLAZE";

    public function __construct($keyId,$applicationKey,$options=[]) {
        $this->setupDebug();
        
        $this->setKeyId($keyId);
        $this->setApplicationKey($applicationKey);
        if(is_null($this->client)) {
            $this->client = new \boru\dhutils\dhHttp();
        }
        $this->initCache(dhGlobal::getDot($options,"cacheFile",""));

        $this->accountInfo = isset($options["accountInfo"]) ? $options["accountInfo"] : [];
        if(empty($this->accountInfo)) {
            $this->loadCache();
        }
        if(($this->getAccount("accountId",false) === false || $this->getAccount("authorizationToken",false) === false) && !is_null($this->getKeyId()) && !is_null($this->getApplicationKey())) {
            $this->authorizeAccount();
        } else {
            $this->_info("Loaded accountInfo from cache ".$this->cacheFile->path());
        }
        if($this->getAccount("accountId",false) === false || $this->getAccount("authorizationToken",false) === false) {
            throw new ClientException(__METHOD__,"Authentication not successful, unable to proceed",-1);
        }
        $this->writeCache();
    }
    protected function initCache($cacheFilePath="") {
        if(empty($cacheFilePath)) {
            $dir = sys_get_temp_dir();
            if(substr($dir,-(strlen(DIRECTORY_SEPARATOR)) != DIRECTORY_SEPARATOR)) {
                $dir.=DIRECTORY_SEPARATOR;
            }
            $cacheFilePath = $dir.'bblazePhp';
        }
        $this->cacheFile = new File([
            "path"=>$cacheFilePath,
            "readMeta"=>true,
            "create"=>true,
            "overwrite"=>false
        ]);
        return $this;
    }
    protected function loadCache() {
        $cached = $this->cacheFile->content();
        if(!empty($cached)) {
            $j = json_decode($cached,true);
            if(is_array($j)) {
                $this->accountInfo = $j;
            }
        }
        return $this;
    }
    protected function writeCache() {
        $this->cacheFile->write(json_encode($this->accountInfo));
        return $this;
    }

    public function authorizeAccount() {
        $this->_trace("funcStart",__METHOD__);
        $response = $this->request("get","/b2_authorize_account",[]);
        if(is_array($response)) {
            $this->accountInfo = $response;
        } else {
            $this->_trace("funcEnd",__METHOD__);
            return false;
        }
        $this->_info("authorizeAccount successful");
        $this->_trace("funcEnd",__METHOD__);
        return true;
    }

    public function createBucket($bucketName,$type=Bucket::T_PRIVATE) {
        $this->_trace("funcStart",__METHOD__);
        $params = [
            "bucketType"=>$type,
            "bucketName"=>$bucketName,
            "accountId"=>$this->getAccount("accountId")
        ];
        $response = $this->request("post","/b2_create_bucket",$params);
        //do something with this.
        if(is_array($response)) {
            $this->_info("createBucket successful",$response["bucketName"]);
            $this->_trace("funcEnd",__METHOD__);
            $bucket = new Bucket($response);
            $this->_bucketCache[$bucket->get("bucketId")] = $bucket;
            return $bucket;
        }
        $this->_trace("funcEnd",__METHOD__);
        return false;
    }
    public function listBuckets($force=false) {
        $this->_trace("funcStart",__METHOD__);
        if(!$force && (time() - $this->_bucketCacheLastList) < $this->bucketCacheTime && !empty($this->_bucketCache)) {
            $this->_trace("funcEnd (cache-hit)",__METHOD__);
            $this->_info("listBuckets (cache) successful",count($this->_bucketCache),"returned");
            return $this->_bucketCache;
        }
        $buckets = [];
        $params = [
            "accountId"=>$this->getAccount("accountId"),
        ];
        $response = $this->request("post","/b2_list_buckets",$params);
        if(is_array($response) && isset($response["buckets"])) {
            foreach($response["buckets"] as $bucket) {
                $buckets[] = new Bucket($bucket);
            }
        }
        $this->_info("listBuckets successful",count($buckets),"returned");
        $this->_trace("funcEnd",__METHOD__);
        $this->_bucketCache = $buckets;
        $this->_bucketCacheLastList = time();
        return $buckets;
    }
    public function getBucketBy($option="bucketName",$value) {
        $this->_trace("funcStart",__METHOD__);
        $buckets = $this->listBuckets();
        foreach($buckets as $bucket) {
            if($bucket->get($option) == $value) {
                $this->_trace("funcEnd",__METHOD__);
                return $bucket;
            }
        }
        $this->_trace("funcEnd",__METHOD__);
        return false;
    }
    public function getBucketFromOptions($options=[]) {
        $this->_trace("funcStart",__METHOD__);
        if(isset($options["bucket"])) {
            $this->_trace("funcEnd",__METHOD__);
           return $options["bucket"];
        } elseif(isset($options["bucketId"])) {
            $this->_trace("funcEnd",__METHOD__);
            return $this->getBucketBy("bucketId",$options["bucketId"]);
        } elseif(isset($options["bucketName"])) {
            $this->_trace("funcEnd",__METHOD__);
            return $this->getBucketBy("bucketName",$options["bucketName"]);
        }
        $this->_trace("funcEnd",__METHOD__);
        return false;
    }

    public function listFiles($options=[]) {
        /**
         * $options = [
         *  "bucket" / "bucketId" / "bucketName",
         *  "fileName" -- to get a file by filename
         *  "maxFileCount" (defaults to 1),
         *  "prefix" -- prefix to search for
         *  "delimiter" -- delimiter to use
         *  "startFileName" -- filename offset to start listing at
         *  "includeDirs" -- defaults to false
         * ]
         */
        $bucket = $this->getBucketFromOptions($options);
        $fileName = dhGlobal::getDot($options,"fileName",null);
        $maxFileCount = dhGlobal::getDot($options,"maxFileCount",1);
        $prefix = dhGlobal::getDot($options,"prefix",null);
        $delimiter = dhGlobal::getDot($options,"delimiter",null);
        $startFileName = dhGlobal::getDot($options,"startFileName",null);
        $includeDirs = dhGlobal::getDot($options,"includeDirs",null);

        $this->_trace("funcStart",__METHOD__);
        $files = [];
        if(!is_null($fileName)) {
            $nextFile = $fileName; 
        } else {
            $nextFile = is_null($startFileName) ? '' : $startFileName;
        }
        while(true) {
            $params = [
                "bucketId"=>$bucket->get("bucketId"),
                "startFileName"=>$nextFile,
                "maxFileCount"=>$maxFileCount,
            ];
            if(!is_null($prefix)) {
                $params["prefix"] = $prefix;
            }
            if(!is_null($delimiter)) {
                $params["delimiter"] = $delimiter;
            }
            $response = $this->request("post","/b2_list_file_names",$params);
            if(is_array($response) && isset($response["files"])) {
                foreach($response["files"] as $file) {
                    if(substr($file['fileName'],-1) == "/" && $delimiter != "/" && !$includeDirs) {
                        continue;
                    }
                    if(!is_null($fileName)) {
                        if($file["fileName"] == $fileName) {
                            $this->listFilesAddToArray($files,new BucketFile($file),$options);
                            break;
                        }
                    } else {
                        $this->listFilesAddToArray($files,new BucketFile($file),$options);
                        //$files[] = new BucketFile($file);
                    }
                }
                if(is_null($response['nextFileName'])) {
                    break;
                } else {
                    print_r($response);
                }
            }
        }
        $this->_trace("funcEnd",__METHOD__);
        return $files;
    }
    protected function listFilesAddToArray(&$array,$bucketFile,$options) {
        $arrayFormat = dhGlobal::getDot($options,"arrayFormat",null);
        if(is_null($arrayFormat) || $arrayFormat == "default" || empty($arrayFormat)) {
            $array[] = $bucketFile;
            return;
        }
        if($arrayFormat == "name") {
            $array[$bucketFile->name()] = $bucketFile;
        }
        if($arrayFormat == "fullName") {
            $array[$bucketFile->fileName()] = $bucketFile;
        }
    }
    public function getFileById($fileId) {
        $this->_trace("funcStart",__METHOD__);
        $params = ['fileId' => $fileId];
        $response = $this->request("post","/b2_get_file_info",$params);
        $this->_trace("funcEnd",__METHOD__);
        return new BucketFile($response);
    }

    public function getFileByName($bucket,$fileName) {
        $this->_trace("funcStart",__METHOD__);
        $files = $this->listFiles($bucket,1,$fileName,null,$fileName);
        foreach($files as $file) {
            if($file->get("fileName") == $fileName) {
                $this->_trace("funcEnd",__METHOD__);
                return $file;
            }
        }
        $this->_trace("funcEnd",__METHOD__);
        return false;
    }

    /**
     * Upload a document to a bucket
     * 
     * the input is an array of options. You can either specify a dhutils File object 'file' or the individual file attributes of 'fileName', 'content', 'contentType' and 'modifiedTime'
     * 
     * If not using the dhutils File object, fileName and content are mandatory. ContentType and modifiedTime are optional. modifiedTime is expected as a unixtimestamp
     * 
     * @param array $options 
     */
    public function upload($options=[]) {
        $this->_trace("funcStart",__METHOD__);
        //bucket, bucketId, bucketName
        //fileName = full path to upload to, no leading /
        //content = file content, or a fopen(filename,'r')
        //contentType (optional)
        //modifiedTime (optional)
        if(($bucket = $this->getBucketFromOptions($options)) === false) {
            throw new ClientException(__METHOD__,"No bucket option provided (bucket, bucketId, bucketName)");
        }
        $file = dhGlobal::getDot($options,"file",null);
        if(!is_null($file) && $file instanceof File) {
            $fileName = dhGlobal::trimString("/",$file->path(),dhGlobal::TRIM_START);
            $modifiedTime = $file->mtime("millis");
            $fileSize = $file->size();
            $fileHash = $file->sha1();
            $contentType = $file->mimeType();
            $content = $file->content();
        } else {
            $modifiedTime = dhGlobal::getDot($options,"modifiedTimeMillis",false);
            if(!$modifiedTime) {
                $mt = dhGlobal::getDot($options,"modifiedTime",false);
                if(is_object($mt)) {
                    $modifiedTime = $mt->format("U")*1000;
                } elseif(is_numeric($mt)) {
                    $modifiedTime = $mt * 1000;
                } else {
                    try {
                        $mt = new \DateTime($mt);
                        $modifiedTime = $mt->format("U")*1000;
                    } catch (\Exception $e) {

                    }
                }
            }
            if(!$modifiedTime) {
                $modifiedTime = floor(microtime(true)*1000);
            }
            //size, //hash
            $fileInfo = dhGlobal::getFileHashAndSize($options["content"]);
            $fileSize = $fileInfo["size"];
            $fileHash = $fileInfo["hash"];
            $fileName = $options["fileName"];
            $contentType = dhGlobal::getDot($options,"contentType","b2/x-auto");
            $content = &$options["content"];
        }
        
        if($fileSize <= $this->uploadNormalLimit) {
            $return = $this->uploadNormal($bucket,$fileName,$contentType,$fileSize,$fileHash,$modifiedTime,$content);
        } else {
            $return = $this->uploadMultiPart($bucket,$fileName,$contentType,$fileSize,$fileHash,$modifiedTime,$content);
        }
        $this->_trace("funcEnd",__METHOD__);
        return $return;
    }
    public function getUploadUrlAuth($bucket,$cache=false) {
        $params = ['bucketId' => $bucket->get("bucketId")];
        $response = $this->request("post","/b2_get_upload_url",$params);
        $this->accountInfo["uploadAuth"] = $response;
        if($cache) {
            $this->writeCache();
        }
        return $response;
    }
    protected function uploadNormal($bucket,$fileName,$contentType,$fileSize,$fileHash,$modifiedTime,$content) {

        $response = $this->getAccount("uploadAuth",null);
        if(is_null($response)) {
            $response = $this->getUploadUrlAuth($bucket,true);
        }
        
        $req = $this->makeRequest("post",$response["uploadUrl"])
            ->authToken($response["authorizationToken"])
            ->header("Content-Type",$contentType)
            ->header("Content-Length",$fileSize)
            ->header("X-Bz-File-Name",$fileName)
            ->header("X-Bz-Content-Sha1",$fileHash)
            ->header("X-Bz-Info-src_last_modified_millis",$modifiedTime)
            ->rawBody($content);
        $this->_trace("funcEnd",__METHOD__);
        $response = $this->sendRequest($req,true);
        if(!is_object($response) && is_array($response)) {
            return new BucketFile($response);
        } else {
            if(is_object($response)) {
                if($response->getStatusCode() == 401 && $response->getReasonPhrase() == "unauthorized") {
                    throw new ClientException(__METHOD__,"401-unauthorized returned from BackBlaze. Account credentials are not valid to perform this action",-1);
                } elseif($response->getStatusCode() == 401) {
                    if($this->uploadTries >= 5) {
                        throw new ClientException(__METHOD__,"upload re-try limit (5) hit");
                    }
                    $this->uploadTries++;
                    $this->accountInfo["uploadAuth"] = null;
                    $retryDelay = $this->uploadDelay * 1000;
                    $this->uploadDelay *= $this->uploadBackoff;
                    usleep($retryDelay);
                    return $this->uploadNormal($bucket,$fileName,$contentType,$fileSize,$fileHash,$modifiedTime,$content);
                } else {
                    throw new ClientException(__METHOD__,$req->getMethod()." Request to ".$req->getUrl()." failed with status code ".$response->getStatusCode()." - ".$response->getReasonPhrase().PHP_EOL.PHP_EOL.$response->body().PHP_EOL,$response->getStatusCode());
                }
            }
        }
    }
    protected function uploadMultiPart($bucket,$fileName,$contentType,$fileSize,$fileHash,$modifiedTime,$content) {
        $this->_trace("funcStart",__METHOD__);
        $params = [
            'bucketId' => $bucket->get("bucketId"),
            'fileName' => $fileName,
            'contentType' => $contentType,
        ];
        $response = $this->request("post","/b2_get_upload_url",$params);
        $newFileId = $response['fileId'];

        $parts = [];
        $partsTotal = ceil($fileSize/$this->multiPartSize);
        for($i=0;$i<$partsTotal;$i++) {
            $sizeSent = $i * $this->mutiPartSize;
            $remainingSize = $fileSize - $sizeSent;
            $size = $remainingSize > $this->multiPartSize ? $this->multiPartSize : $remainingSize;

            $partInfo = dhGlobal::getFileHashAndSize($content,$sizeSent,$size);
            $partSize = $partInfo["size"];
            $partHash = $partInfo["hash"];
            $parts[] = $partHash;

            $response = $this->request("post","/b2_get_upload_part_url",["fileId"=>$newFileId]);
            $request = $this->makeRequest("post",$response["uploadUrl"])
                ->authToken($response["authorizationToken"])
                ->header("X-Bz-Part-Number",$i+1)
                ->header("Content-Length",$partSize)
                ->header("X-Bz-Content-Sha1",$partHash)
                ->rawBody(dhGlobal::getPartOfFile($content,$sizeSent,$size));
            $response = $this->sendRequest($request,true);
        }

        $response = $this->request("post","/b2_finish_large_file",["fileId"=>$newFileId,"partSha1Array"=>$parts]);
        $this->_trace("funcEnd",__METHOD__);
        return new BucketFile($response);
    }


    public function makeRequest($method,$uri='') {
        $this->_trace("funcStart",__METHOD__);
        $methods = ["get","post","put","delete","options"];
        if(substr($uri,0,4) != "http") {
            if(substr($uri,0,1) != "/") {
                $uri = "/".$uri;
            }
            $url = $this->getAccount("apiUrl",$this->apiUrl);

            $url .= $this->apiVersion.$uri;
        } else {
            $url = $uri;
        }
        if(!in_array(strtolower($method),$methods)) {
            throw new ClientException(__METHOD__,"HTTP Method $method not valid, options are [".implode(",",$methods)."]");
        }
        $method = strtoupper($method);
        $this->_trace("funcEnd",__METHOD__);
        return $this->client->request($method,$url);
    }
    public function sendRequest($request,$json=true,$options=[]) {
        $this->_trace("funcStart",__METHOD__);
        try {
            $response = $request->send();
        } catch(\Exception $e) {
            throw new ClientException(__METHOD__,$e->getMessage(),$e->getCode());
        }
        if($response->getStatusCode() === 200) {
            if($json) {
                $this->_trace("funcEnd",__METHOD__);
                return $response->body(true,true);
            } else {
                $this->_trace("funcEnd",__METHOD__);
                return $response->body();
            }
        } else {
            if(!empty($options) && isset($options["returnError"])) {
                return $response;
            } else {
                throw new ClientException(__METHOD__,$request->getMethod()." Request to ".$request->getUrl()." failed with status code ".$response->getStatusCode()." - ".$response->getReasonPhrase().PHP_EOL.PHP_EOL.$response->body().PHP_EOL,$response->getStatusCode());
            }
        }
    }
    public function request($method,$uri='',$params=[],$json=true) {
        $this->_trace("funcStart",__METHOD__);
        $request = $this->makeRequest($method,$uri);
        if(isset($this->accountInfo["authorizationToken"])) {
            $request->authToken($this->accountInfo["authorizationToken"]);
        } else {
            $request->authBasic($this->keyId,$this->applicationKey);
        }
        if(!empty($params)) {
            if(is_object($params)) {
                $arr = $params->get();
                foreach($arr as $k=>$v) {
                    if(method_exists($request,$k)) {
                        call_user_func([$request,$k],$v);
                    }
                }
            }
            if(is_array($params)) {
                foreach($params as $k=>$v) {
                    $request->json($k,$v);
                }
            }
        }
        $this->_trace("funcEnd",__METHOD__);
        return $this->sendRequest($request,$json);
    }

    public function getAccount($item=null,$default=false) {
        $this->_trace("funcStart",__METHOD__);
        if(is_null($this->accountInfo)) {
            $this->_trace("funcEnd",__METHOD__);
            return $default;
        }
        if(is_null($item)) {
            $this->_trace("funcEnd",__METHOD__);
            return $this->accountInfo;
        } elseif(!isset($this->accountInfo[$item])) {
            $this->_trace("funcEnd",__METHOD__);
            return $default;
        }
        $this->_trace("funcEnd",__METHOD__);
        return $this->accountInfo[$item];
    }

    /**
     * Get the value of keyId
     */ 
    public function getKeyId()
    {
        return $this->keyId;
    }

    /**
     * Set the value of keyId
     *
     * @return  self
     */ 
    public function setKeyId($keyId)
    {
        $this->keyId = $keyId;

        return $this;
    }

    /**
     * Get the value of applicationKey
     */ 
    public function getApplicationKey()
    {
        return $this->applicationKey;
    }

    /**
     * Set the value of applicationKey
     *
     * @return  self
     */ 
    public function setApplicationKey($applicationKey)
    {
        $this->applicationKey = $applicationKey;

        return $this;
    }
}



class ClientException extends BackBlazeBaseException {
    protected $debug_name = "backblaze";
}