<?php
namespace boru\backblaze;

use \boru\backblaze\parts\traits\TraitLog;

use \boru\backblaze\Client;
use \boru\backblaze\parts\BucketFile;
use boru\dhutils\dhGlobal;
use InvalidArgumentException;
use Exception;
use React\Promise\Deferred;
use UnexpectedValueException;

class Files {
    use TraitLog;

    protected $_bucketCacheLastList = 0;
    protected $_bucketCache = [];

    protected $debug_name = "backblaze";
    protected $debug_prefix = "BBBucket";
    public $debugDoTrace = false;

    /** @var Client */
    protected $client;

    public function __construct($clientOrOptions) {
        if(is_object($clientOrOptions) && $clientOrOptions instanceof Client) {
            $this->client = $clientOrOptions;
        } elseif(is_array($clientOrOptions)) {
            $this->client = Client::fromAuthArray($clientOrOptions);
        }
        if(is_null($this->client)) {
            throw new \Exception(__CLASS__." - failed constructor, client or auth array invalid");
        }
    }

    /**
     * 
     * @param mixed $fileId 
     * @param bool $async 
     * @return string|\React\Promise\ExtendedPromiseInterface
     * @throws InvalidArgumentException 
     * @throws Exception 
     * @throws UnexpectedValueException 
     */
    public function downloadById($fileId,$headersOnly=false,$async=false) {
        $request = $this->client->request("get","/b2_download_file_by_id?fileId=$fileId",[]);
        if($headersOnly) {
            $request->header("Range","bytes=0-1");
        }
        if($async) {
            $deferred = new Deferred();
            $request->sendAsync()->then(function($response) use (&$deferred) {
                $deferred->resolve($response);
            },function($e) use (&$deferred) {
                $deferred->reject($e);
            });
            return $deferred->promise();
        } else {
            return $request->send();
        }
        //return 
    }

    /** @return \boru\dhutils\http\Response|\React\Promise\ExtendedPromiseInterface */
    public function downloadByName($fileName,$headersOnly=false,$async=false) {
        $url = $this->client->getDownloadUrl();
        if(!empty($url)) {
            $url.="/file";
            $url.="/".$fileName;
            $request = $this->client->request("get",$url);
            if($headersOnly) {
                $request->header("Range","bytes=0-1");
            }
            if($async) {
                $deferred = new Deferred();
                $request->sendAsync()->then(function($response) use (&$deferred) {
                    $deferred->resolve($response);
                },function($e) use (&$deferred) {
                    $deferred->reject($e);
                });
                return $deferred->promise();
            }
            return $request->send();
        }
        return false;
    }

    public function getById($fileId) {
        $this->_trace("funcStart",__METHOD__);
        $params = ['fileId' => $fileId];
        $request = $this->client->request("post","/b2_get_file_info",$params);
        if(($response = $request->jsonParseSend()) !== false) {
            $this->_trace("funcEnd",__METHOD__);
            return new BucketFile($response);
        }
    }
    public function getByName($bucket,$fileName,$dir=false) {
        $this->_trace("funcStart",__METHOD__);
        $prefix = $fileName;
        if($dir) {
            if(substr($prefix,-1) != "/") {
                $fileName.="/";
            } else {
                $prefix = substr($prefix,0,-1);
            }
        }
        /*
        $pathWithSlash(filename) = $path(prefix);
        if(substr($path,-1) != "/") {
            $pathWithSlash.="/";
        } else {
            $path = substr($path,0,-1);
        }*/
        $options = [
            "maxFileCount"=>1,
            "prefix"=>$prefix,
            "fileName"=>$fileName,
            "delimiter"=>"/"
        ];
        if($dir) {
            $options["includeDirs"] = true;
        }
        if(is_object($bucket)) {
            $options["bucket"] = $bucket;
        } else {
            $options["bucketId"] = $bucket;
        }
        $files = $this->listFiles($options);
        //$bucket,1,$fileName,null,$fileName);
        //print_r($files);
        foreach($files as $file) {
            if($file->get("fileName") == $fileName) {
                $this->_trace("funcEnd",__METHOD__);
                return $file;
            }
        }
        $this->_trace("funcEnd",__METHOD__);
        return false;
    }

    /**
     * 
     * @param array $options 
     * @return false|BucketFile[] 
     */
    public function listFiles($options=[],$versions=false) {
        /**
         * $options = [
         *  "bucket" / "bucketId" / "bucketName",
         *  "fileName" -- to get a file by filename
         *  "maxFileCount" (defaults to 1),
         *  "prefix" -- prefix to search for
         *  "delimiter" -- delimiter to use
         *  "startFileName" -- filename offset to start listing at
         *  "includeDirs" -- defaults to false
         * ]
         */
        if(($bucketId = dhGlobal::getDot($options,"bucketId",false))!==false) {

        } elseif(($bucket = $this->client->buckets()->fromOptions($options)) !== false) {
            $bucketId = $bucket->get("bucketId");
        }
        $fileName        = dhGlobal::getDot($options,"fileName",null);
        $maxFileCount    = dhGlobal::getDot($options,"maxFileCount",1);
        $prefix          = dhGlobal::getDot($options,"prefix",null);
        $delimiter       = dhGlobal::getDot($options,"delimiter",null);
        $startFileName   = dhGlobal::getDot($options,"startFileName",null);
        if($versions) {
            $maxIterations   = dhGlobal::getDot($options,"maxIterations", $versions ? 5 : 3);
            $filesPerRequest = dhGlobal::getDot($options,"filesPerRequest",1000);
            $options["isVersionsRequest"] = $versions;
        } else {
            $maxIterations   = dhGlobal::getDot($options,"maxIterations", 3);
            $filesPerRequest = dhGlobal::getDot($options,"filesPerRequest",is_null($fileName) ? 1000 : 1);
        }
        

        $this->_trace("funcStart",__METHOD__);
        $files = [];
        if(!is_null($fileName)) {
            $nextFile = $fileName; 
        } else {
            $nextFile = is_null($startFileName) ? '' : $startFileName;
        }
        $continue = true;
        $iterations = 0;
        while($continue) {
            $iterations++;
            $params = $this->makeListParams($bucketId,$nextFile,$filesPerRequest,$prefix,$delimiter);
            
            if($versions) {
                $request = $this->client->request("post","/b2_list_file_versions",$params);
            } else {
                $request = $this->client->request("post","/b2_list_file_names",$params);
            }
            if(($response = $request->jsonParseSend()) !== false && isset($response["files"])) {

                $continue = $this->parseListResponseFile($files,$response,$maxFileCount,$delimiter,$fileName,$options);

                if(is_null($response['nextFileName'])) {
                    $continue = false;
                    break;
                } else {
                    $nextFile = $response["nextFileName"];
                }
            } else {
                return false;
            }
            if($maxIterations>0 && $iterations>=$maxIterations) {
                break;
            }
        }
        $this->_trace("funcEnd",__METHOD__);
        return $files;
    }
    private function parseListResponseFile(&$files,$response,$maxFileCount,$delimiter="",$fileName=null,$options=[]) {
        $includeDirs      = dhGlobal::getDot($options,"includeDirs",false);
        $isVersionRequest = dhGlobal::getDot($options,"isVersionsRequest",false);
        foreach($response["files"] as $file) {
            if(count($files) >= $maxFileCount) {
                return false;
            }
            $isDir = substr($file['fileName'],-1) == "/" ? true : false;
            if($isDir && $delimiter != "/" && !$includeDirs) {
                continue;
            }
            if(!is_null($fileName)) {
                if($file["fileName"] == $fileName) {
                    $this->listFilesAddToArray($files,new BucketFile($file),$options);
                    if(!$isVersionRequest) {
                        return false;
                    }
                } elseif($includeDirs && substr($file['fileName'],-1) == "/" && substr($file['fileName'],0,-1) == $fileName) {
                    $this->listFilesAddToArray($files,new BucketFile($file),$options);
                    if(!$isVersionRequest) {
                        return false;
                    }
                }
            } else {
                $this->listFilesAddToArray($files,new BucketFile($file),$options);
            }
        }
        return true;
    }
    protected function listFilesAddToArray(&$array,$bucketFile,$options) {
        $arrayFormat = dhGlobal::getDot($options,"arrayFormat",null);
        if(is_null($arrayFormat) || $arrayFormat == "default" || empty($arrayFormat)) {
            $array[] = $bucketFile;
            return;
        }
        if($arrayFormat == "name") {
            $array[$bucketFile->name()] = $bucketFile;
        }
        if($arrayFormat == "fullName") {
            $array[$bucketFile->fileName()] = $bucketFile;
        }
    }

    /** @return BucketFile[] */
    public function versions($options=[]) {
        return $this->listFiles($options,true);
    }

    private function makeListParams($bucketId,$nextFile,$filesPerRequest,$prefix,$delimiter) {
        $params = [
            "bucketId"=>$bucketId,
            "startFileName"=>$nextFile,
            "maxFileCount"=>$filesPerRequest,
        ];
        if(!is_null($prefix)) {
            $params["prefix"] = $prefix;
        }
        if(!is_null($delimiter)) {
            $params["delimiter"] = $delimiter;
        }
        return $params;
    }
}