<?php
namespace boru\backblaze;

use \boru\backblaze\BackBlazeBase;
use \boru\backblaze\BackBlazeBaseException;
use \boru\dhutils\dhGlobal;
use \boru\backblaze\Client;
use \boru\dhutils\filesys\Directory;

class Path extends BackBlazeBase {
    protected $client;
    protected $localDir;
    protected $bucket;
    protected $bucketPath;

    protected $debug_name = "path";
    protected $debug_prefix = "BBPATH";


    protected $directories = [];
    protected $files = [];
    protected $changed = [];
    protected $uploaded = [];

    public function __construct($options=[]) {
        $this->setupDebug();

        if(isset($options["localDirectory"]) && !empty($options["localDirectory"])) {
            $this->localDir = new Directory([
                "path"=>$options["localDirectory"],
                "scan"=>true,
                "recursive"=>true
            ]);
        } elseif(isset($options["localDir"]) && !empty($options["localDir"])) {
            $this->localDir = new Directory([
                "path"=>$options["localDir"],
                "scan"=>true,
                "recursive"=>true
            ]);
        }
        if(is_null($this->localDir)) {
            throw new PathException(__METHOD__,"local directory not specified in options (localDir or localDirectory)",-1);
        }

        $this->bucketPath = dhGlobal::getDot($options,"bucketPath",null);
        if(is_null($this->bucketPath)) {
            throw new PathException(__METHOD__,"remote bucketPath not specified in options (bucketPath)",-1);
        }
        if(substr($this->bucketPath,-1) != "/") {
            $this->bucketPath.="/";
        }
        if(substr($this->bucketPath,0,1) == "/") {
            if($this->bucketPath == "/") {
                $this->bucketPath = "";
            } else {
                $this->bucketPath = substr($this->bucketPath,1);
            }
        }

        $this->setClientFromOptions($options);
        $this->setBucketFromOptions($options);
    }

    public function getStats() {
        $stats = [
            "directories"=>count($this->directories),
            "files"=>0,
            "size"=>0,
            "changed"=>[
                "files"=>0,
                "size"=>0,
            ],
            "uploaded"=>[
                "files"=>0,
                "size"=>0
            ]
        ];
        foreach($this->files as $file) {
            $stats["files"]++;
            $stats["size"]+=$file->size();
            $changed = $file->get("changed",false);
            $uploaded = $file->get("uploaded",false);
            if($changed) {
                $stats["changed"]["files"]++;
                $stats["changed"]["size"]+=$file->size();
            }
            if($uploaded) {
                $stats["uploaded"]["files"]++;
                $stats["uploaded"]["size"]+=$file->size();
            }
        }
        return $stats;
    }
    public function files() {
        return $this->files;
    }
    public function dirs() {
        return $this->dirs;
    }

    public function scan(callable $fileComplete=null,callable $directoryComplete=null) {
        $this->_info("scan started");
        $this->syncDirectory($this->localDir,true,false,$fileComplete,$directoryComplete);
    }
    public function sync($forceFullSync=false,callable $fileComplete=null,callable $directoryComplete=null) {
        $this->_info("sync started");
        $this->syncDirectory($this->localDir,false,$forceFullSync,$fileComplete,$directoryComplete);
    }
    protected function syncDirectory($Directory,$scanOnly=false,$forceFullSync=false,callable $fileComplete=null,callable $directoryComplete=null) {
        //Get backblaze files and info for the parent directory..
        $bdir = $this->bucketPath.substr($Directory->path(),1)."/";
        $files = $Directory->files();
        $dirs = $Directory->dirs();
        $fileCount = is_array($files) ? count($files) : 0;
        $dirCount = is_array($dirs) ? count($dirs) : 0;
        $bbFiles = $this->client->listFiles([
            "bucket"=>$this->bucket,
            "maxFileCount"=>1000,
            "delimiter"=>"/",
            "prefix"=>$bdir,
            "arrayFormat"=>"name"
        ]);
        $bbFileCount = is_array($bbFiles) ? count($bbFiles) : 0;

        $this->_info("Starting ".$bdir." - Files:".$fileCount." Subdirs:".$dirCount." BackBlaze:".$bbFileCount." ");
        
        $filesProcessed = [];
        if(!empty($files) && is_array($files)) {
            $count=0;
            
            
            foreach($files as $file) {
                $this->files[$file->path()] = $file;
                $fRef = &$this->files[$file->path()];
                $count++;
                $name = $file->name();
                $bbFile = isset($bbFiles[$name]) ? $bbFiles[$name] : false;
                $fRef->set("backblazeFile",$bbFile);
                $changed = true;
                if(!$forceFullSync) {
                    if($bbFile !== false) {
                        if(strlen($file->sha1())>10) {
                            if($file->sha1() == $bbFile->sha1()) {
                                //$this->_debug("sha match on",$file->path());
                                $changed = false;
                            }
                        }
                    }
                }
                $uploaded = false;
                if($changed) {
                    $this->changed[] = &$this->files[$file->path()];
                    if(!$scanOnly) {
                        try {
                            $this->client->upload([
                                "bucket"=>$this->bucket,
                                "file"=>$file
                            ]);
                            $uploaded = true;
                            $this->uploaded[] = &$this->files[$file->path()];
                        } catch (\Exception $e) {

                        }
                    }
                }
                $fRef->set("changed",$changed);
                $fRef->set("uploaded",$uploaded);
                $filesProcessed[] = &$this->files[$file->path()];
                if(is_callable($fileComplete)) {
                    $fileComplete([
                        "sync"=>$this,
                        "file"=>$fRef,
                        "num"=>$count-1,
                        "fileCount"=>$fileCount,
                        "bbFileCount"=>$bbFileCount
                    ]);
                } else {
                    $this->_trace("Processed ".$fRef->path()." - ".(
                        $changed ? 
                            ($uploaded ? "Uploaded" : "Changed")
                            : "Skipped"
                        )
                    );
                }
            }
            $file = null;
            $bbFile = null;
            unset($fRef);
        }
        if(is_callable($directoryComplete)) {
            $directoryComplete([
                "sync"=>$this,
                "directory"=>$Directory,
                "processed"=>$filesProcessed,
                "fileCount"=>$fileCount,
                "bbFileCount"=>$bbFileCount
            ]);
        }
        $Directory->set("filesProcessed",$filesProcessed);
        $Directory->set("fileCount",$fileCount);
        $Directory->set("bbFileCount",$bbFileCount);
        $this->directories[$Directory->path()] = $Directory;
        $this->_info("complete with ".$Directory->path()." .. processed ".count($filesProcessed)." files");
        unset($filesProcessed);
        if(!empty($dirs) && is_array($dirs)) {
            foreach($dirs as $dir) {
                $this->syncDirectory($dir,$scanOnly,$forceFullSync,$fileComplete,$directoryComplete);
            }
        }
    }

    public function setClientFromOptions($options=[]) {
        if(isset($options["client"])) {
            $this->client = $options["client"];
            return $this->client;
        }
        $keyId = dhGlobal::getDot($options,"keyId",null);
        $applicationKey = dhGlobal::getDot($options,"applicationKey",null);
        $accountInfo = dhGlobal::getDot($options,"accountInfo",[]);
        $options["accountInfo"] = $accountInfo;
        $cacheFile = dhGlobal::getDot($options,"cacheFile",false);
        if($cacheFile !== false) {
            $options["cacheFile"] = $cacheFile;
        }
        $this->client = new Client($keyId,$applicationKey,$options);
        return $this->client;
    }

    public function setBucketFromOptions($options=[]) {
        $this->_trace("funcStart",__METHOD__);
        if(isset($options["bucket"])) {
           $this->bucket = $options["bucket"];
        } elseif(isset($options["bucketId"])) {
            $this->bucket = $this->client->getBucketBy("bucketId",$options["bucketId"]);
        } elseif(isset($options["bucketName"])) {
            $this->bucket = $this->client->getBucketBy("bucketName",$options["bucketName"]);
        }
        $this->_trace("funcEnd",__METHOD__);
        if(is_null($this->bucket)) {
            return false;
        }
        return $this->bucket;
    }


    
}

class PathException extends BackBlazeBaseException {
    protected $debug_name = "backblaze-path";
}