<?php
namespace boru\backblaze;

use boru\backblaze\parts\traits\TraitLog;
use \boru\backblaze\Client;
use boru\backblaze\exceptions\ClientException;
use \boru\backblaze\parts\BucketFile;
use boru\dhutils\dhGlobal;
use boru\dhutils\filesys\File;
use Exception;
use InvalidArgumentException;
use React\Promise\Deferred;
use React\Promise\PromiseInterface;
use UnexpectedValueException;

class Upload {
    public function setupDebug() {
        dhGlobal::addLogLevel($this->debug_name."-t","trace","TRACE"); //_trace()
        dhGlobal::addLogLevel($this->debug_name."-d","debug","DEBUG"); //_debug()
        dhGlobal::addLogLevel($this->debug_name."-i","info"," INFO"); //_info()
        dhGlobal::addLogLevel($this->debug_name."-w","warn"," WARN"); //_warn()
        dhGlobal::addLogLevel($this->debug_name."-e","error","ERROR"); //_error()
    }

    protected function _trace(...$args) {
        if($this->debugDoTrace) {
            array_unshift($args,$this->debug_prefix." - ");
            array_unshift($args,$this->debug_name."-t");
            return call_user_func_array([$this,"log"],$args);
        }
        return;
    }
    protected function _debug(...$args) {
        array_unshift($args,$this->debug_prefix." - ");
        array_unshift($args,$this->debug_name."-d");
        return call_user_func_array([$this,"log"],$args);
    }
    protected function _info(...$args) {
        array_unshift($args,$this->debug_prefix." - ");
        array_unshift($args,$this->debug_name."-i");
        return call_user_func_array([$this,"log"],$args);
    }
    protected function _warn(...$args) {
        array_unshift($args,$this->debug_prefix." - ");
        array_unshift($args,$this->debug_name."-w");
        return call_user_func_array([$this,"log"],$args);
    }
    protected function _error(...$args) {
        array_unshift($args,$this->debug_prefix." - ");
        array_unshift($args,$this->debug_name."-e");
        return call_user_func_array([$this,"log"],$args);
    }
    public function log(...$args) {
        return dhGlobal::log(...$args);
    }

    protected $_bucketCacheLastList = 0;
    protected $_bucketCache = [];

    protected $debug_name = "backblaze";
    protected $debug_prefix = "BBUPLOAD";
    public $debugDoTrace = false;

    protected $async = true;

    public $uploadNormalLimit = 3000000000; //3gb
    public $multiPartSize = 100000000; //100mb
    //public $uploadNormalLimit = 100000000; //100mb
    //public $multiPartSize = 20000000; //20MB

    protected $bucketId;
    protected $fileName;
    protected $contentType;
    protected $fileSize;
    protected $fileHash;
    protected $modifiedTime;
    protected $content;
    protected $meta;
    protected $partPieces = [];

    /** @var Client */
    protected $client;

    public function __construct($clientOrOptions) {
        if(is_object($clientOrOptions) && $clientOrOptions instanceof Client) {
            $this->client = $clientOrOptions;
        } elseif(is_array($clientOrOptions)) {
            $this->client = Client::fromAuthArray($clientOrOptions);
        }
        if(is_null($this->client)) {
            throw new \Exception(__CLASS__." - failed constructor, client or auth array invalid");
        }
    }

    /**
     * 
     * @param string|Bucket $bucket 
     * @param bool $cache 
     * @return \React\Promise\Promise 
     * @throws ClientException 
     * @throws ClientException 
     */
    public function getUploadUrlAuth($bucket,$cache=false) {
        $this->_trace("funcStart",__METHOD__);
        $deferred = new \React\Promise\Deferred();
        if(is_object($bucket)) {
            $params = ['bucketId' => $bucket->get("bucketId")];
        } else {
            $params = ['bucketId' => $bucket];
        }
        $response = false;
        if($cache) {
            $response = $this->client->getUploadAuth();
        }
        if($response !== false && $cache) {
            $deferred->resolve($response);
        } else {
            $request = $this->client->request("post","/b2_get_upload_url",$params);
            $request->sendAsync(function($response) use ($deferred,$cache) {
                $json = $response->body(true);
                if(is_array($json)) {
                    $this->client->accountSet("uploadAuth",$json,$cache);
                }
                $this->_trace("funcEnd",__METHOD__,__LINE__);
                $deferred->resolve($json);
            },function($e) use ($deferred,$cache) {
                $this->_trace("funcEnd with error",__METHOD__,__LINE__);
                $deferred->reject($e);
                //throw new \Exception("b2_get_upload_url failed");
            });
        }
        return $deferred->promise();
    }

    /**
     * 
     * @param array $options 
     * @return false|BucketFile|PromiseInterface 
     * @throws InvalidArgumentException 
     * @throws Exception 
     * @throws UnexpectedValueException 
     * @throws ClientException 
     */
    public function upload($options=[]) {
        
        $this->_trace("funcStart",__METHOD__);
        $bucketId = false;
        if(($bucketId = dhGlobal::getDot($options,"bucketId",false))!==false) {

        } elseif(($bucket = $this->client->buckets()->fromOptions($options)) !== false) {
            $bucketId = $bucket->get("bucketId");
        } else {
            throw new ClientException(__METHOD__,"No bucket option provided (bucket, bucketId, bucketName)");
        }
        $this->bucketId = $bucketId;

        $this->async=true;
        if(($async=dhGlobal::getVal($options,"async",false)) !== false) {
            $this->async = true;
        }
        $timeout = dhGlobal::getVal($options,"timeout",true);

        $uploadNormalLimit = dhGlobal::getVal($options,"uploadNormalLimit",$this->uploadNormalLimit);
        $multiPartSize = dhGlobal::getVal($options,"multiPartSize",$this->multiPartSize); 

        $file = dhGlobal::getDot($options,"file",null);
        if(!is_null($file) && $file instanceof File) {
            $prefix = dhGlobal::trimString("/",dhGlobal::getDot($options,"prefix",""),dhGlobal::TRIM_BOTH);
            if(($fileName = dhGlobal::getVal($options,"fileName",false)) !== false) {
                $this->fileName = $fileName;
            } else {
                $this->fileName = dhGlobal::trimString("/",$prefix.$file->path(),dhGlobal::TRIM_START);
            }
            
            $this->modifiedTime = $file->mtime("millis");
            $this->fileSize = $file->size();
            $this->fileHash = $file->sha1();
            $this->contentType = $file->mimeType();
            $this->content = $file->content();
            $file=null;
        } else {
            $this->modifiedTime = dhGlobal::getDot($options,"modifiedTimeMillis",false);
            if(!$this->modifiedTime) {
                $mt = dhGlobal::getDot($options,"modifiedTime",false);
                if(is_object($mt)) {
                    $this->modifiedTime = $mt->format("U")*1000;
                } elseif(is_numeric($mt)) {
                    $this->modifiedTime = $mt * 1000;
                } else {
                    try {
                        $mt = new \DateTime($mt);
                        $this->modifiedTime = $mt->format("U")*1000;
                    } catch (\Exception $e) {

                    }
                }
            }
            if(!$this->modifiedTime) {
                $this->modifiedTime = floor(microtime(true)*1000);
            }
            //size, //hash
            $fileInfo = dhGlobal::getFileHashAndSize($options["content"]);
            $this->fileSize = $fileInfo["size"];
            $this->fileHash = $fileInfo["hash"];
            $this->fileName = $options["fileName"];
            $this->contentType = dhGlobal::getDot($options,"contentType","b2/x-auto");
            $this->content = &$options["content"];
        }
        $this->meta = dhGlobal::getDot($options,"meta",[]);
        
        if($this->fileSize <= $uploadNormalLimit) {
            $return = $this->uploadNormal($timeout);
        } else {
            if($this->async) {
                $return = $this->uploadMultiPartAsync($timeout,$multiPartSize);
            } else {
                $return = $this->uploadMultiPart($timeout,$multiPartSize);
            }
        }
        $this->_trace("funcEnd",__METHOD__);
        
        
        if($return instanceof PromiseInterface) {
            $return->then(function($resp) {
                
                
                
            });
        }
        return $return;
    }

    /**
     * 
     * @return PromiseInterface 
     * @throws ClientException 
     */
    protected function uploadNormal($timeout=true) {
        $this->_trace("funcStart",__METHOD__);
        if($this->async) {
            $cache = false;
        } else {
            $cache = true;
        }
        $uDeferred = new \React\Promise\Deferred();
        $this->getUploadUrlAuth($this->bucketId,false)->then(function($value) use (&$uDeferred,$timeout) {
            $request = $this->client->request("post",$value["uploadUrl"]);
            $request->authToken($value["authorizationToken"])
                ->header("Content-Type",$this->contentType)
                ->header("Content-Length",$this->fileSize)
                ->header("X-Bz-File-Name",$this->fileName)
                ->header("X-Bz-Content-Sha1",$this->fileHash)
                ->header("X-Bz-Info-src_last_modified_millis",$this->modifiedTime)
                ->rawBody($this->content);
            if(!empty($this->meta)) {
                foreach($this->meta as $k=>$v) {
                    $request->header("X-Bz-Info-".$k,$v);
                }
            }
            
            if($this->async) {
                $request->async(true);
                $request->sendAsync(function($response) use ($request,&$uDeferred,$timeout) {
                    $this->_trace("funcEnd",__METHOD__);
                    $this->handleNormalReponse($request,$response,$uDeferred,$timeout);
                },function($e) use ($request,&$uDeferred) {
                    $this->_trace("funcEnd",__METHOD__);
                    if(!is_null($uDeferred)) {
                        $uDeferred->reject($request->getMethod()." Request to ".$request->getUrl()." failed with exception ".$e->getCode()." - ".$e->getMessage());
                    } else {
                        throw new ClientException(__METHOD__,$request->getMethod()." Request to ".$request->getUrl()." failed with exception ".$e->getCode()." - ".$e->getMessage());
                    }
                },$timeout);
            } else {
                $response = $request->send();
                $this->handleNormalReponse($request,$response,$uDeferred,$timeout);
                $this->_trace("funcEnd",__METHOD__);
            }
        },function($e) use (&$uDeferred) {
            $this->_trace("funcEnd",__METHOD__);
            if(!is_null($uDeferred)) {
                $uDeferred->reject($e);
            } else {
                throw new \Exception($e->getMessage(),$e->getCode());   
            }
        });
        
        //if(!$this->async) {
        //    return \React\Async\await($uDeferred->promise());
        //}
        return $uDeferred->promise();        
    }
    protected function handleNormalReponse($request,$response,$deferred=null,$timeout=true) {
        $this->_trace("funcStart",__METHOD__);
        $json = $response->body(true);
        if(is_array($json)) {
            $bucketFile = new BucketFile($json);
            $bucketFile->set("sourceSha1",$this->fileHash);
            dhGlobal::debug("uploaded ".$bucketFile->fileName(),$bucketFile->fileId());
            dhGlobal::increment("bb.async.completed");
            if(!is_null($deferred)) {
                $deferred->resolve($bucketFile);
            }
            $this->_trace("funcEnd",__METHOD__);
            return $bucketFile;
        } else {
            if(is_object($response)) {
                if($response->getStatusCode() == 401 && $response->getReasonPhrase() == "unauthorized") {
                    $this->_trace("funcEnd",__METHOD__);
                    if(!is_null($deferred)) {
                        $deferred->reject("401-unauthorized returned from BackBlaze. Account credentials are not valid to perform this action");
                    } else {
                        throw new ClientException(__METHOD__,"401-unauthorized returned from BackBlaze. Account credentials are not valid to perform this action",-1);
                    }
                } elseif($response->getStatusCode() == 401 || $response->getStatusCode() == 400) {
                    if($this->uploadTries >= 5) {
                        $this->_trace("funcEnd",__METHOD__);
                        if(!is_null($deferred)) {
                            $deferred->reject("upload re-try limit (5) hit");
                        } else {
                            throw new ClientException(__METHOD__,"upload re-try limit (5) hit");
                        }
                    }
                    $this->uploadTries++;
                    $this->accountInfo["uploadAuth"] = null;
                    $retryDelay = $this->uploadDelay * 1000;
                    $this->uploadDelay *= $this->uploadBackoff;
                    usleep($retryDelay);
                    $this->_trace("funcEnd",__METHOD__);
                    return $this->uploadNormal($timeout);
                } else {
                    $this->_trace("funcEnd",__METHOD__);
                    if(!is_null($deferred)) {
                        $deferred->reject($request->getMethod()." Request to ".$request->getUrl()." failed with status code ".$response->getStatusCode()." - ".$response->getReasonPhrase().PHP_EOL.PHP_EOL.$response->body().PHP_EOL);
                    } else {
                        throw new ClientException(__METHOD__,$request->getMethod()." Request to ".$request->getUrl()." failed with status code ".$response->getStatusCode()." - ".$response->getReasonPhrase().PHP_EOL.PHP_EOL.$response->body().PHP_EOL,$response->getStatusCode());
                    }
                }
            } else {
                if(is_array($response)) {
                    $response = json_encode($response);
                }
                $this->_trace("funcEnd",__METHOD__);
                if(!is_null($deferred)) {
                    $deferred->reject($request->getMethod()." Request to ".$request->getUrl()." failed with non-object: ".$response);
                } else {
                    throw new \Exception("HTTP Response to ".$request->getMethod()." Request to ".$request->getUrl()." is not an object",-1);
                }
            }
        }
    }


    /**
     * We still return a promiss, even tho it's synchronous
     * @param bool $timeout 
     * @return false|BucketFile 
     */
    protected function uploadMultiPart($timeout=true,$multiPartSize=null) {
        
        if(is_null($multiPartSize)) {
            $multiPartSize = $this->multiPartSize;
        }
        $uDeferred = new \React\Promise\Deferred();
        $this->_trace("funcStart",__METHOD__);
        $params = [
            'bucketId' => $this->bucketId,
            'fileName' => $this->fileName,
            'contentType' => $this->contentType,
            'fileInfo' => [
                "large_file_sha1" => $this->fileHash,
                "src_last_modified_millis"=> $this->modifiedTime
            ]
        ];
        if(!empty($meta)) {
            $params["fileInfo"] = $meta;
        }
        $request = $this->client->request("post","/b2_start_large_file",$params);
        if(($response = $request->jsonParseSend()) !== false) {
            //print_r($response);
            $newFileId = $response['fileId'];
        } else {
            $uDeferred->reject($response);
            return $uDeferred->promise();
        }
        
        $parts = [];
        $partsTotal = ceil($this->fileSize/$multiPartSize);
        for($i=0;$i<$partsTotal;$i++) {
            dhGlobal::outLine("part $i");
            $sizeSent = $i * $multiPartSize;
            $remainingSize = $this->fileSize - $sizeSent;
            $size = $remainingSize > $multiPartSize ? $multiPartSize : $remainingSize;

            $partInfo = dhGlobal::getFileHashAndSize($this->content,$sizeSent,$size);
            $partSize = $partInfo["size"];
            $partHash = $partInfo["hash"];
            $parts[] = $partHash;

            $request = $this->client->request("post","/b2_get_upload_part_url",["fileId"=>$newFileId]);
            if(($response = $request->jsonParseSend()) !== false) {
                print_r($response);
                $request = $request = $this->client->request("post",$response["uploadUrl"])
                    ->authToken($response["authorizationToken"])
                    ->header("X-Bz-Part-Number",$i+1)
                    ->header("Content-Length",$partSize)
                    ->header("X-Bz-Content-Sha1",$partHash)
                    ->rawBody(dhGlobal::getPartOfFile($this->content,$sizeSent,$size));
                $response = $request->send();
            }
            
        }

        $request = $this->client->request("post","/b2_finish_large_file",["fileId"=>$newFileId,"partSha1Array"=>$parts]);
        
        $this->_trace("funcEnd",__METHOD__);
        if(($response = $request->jsonParseSend()) !== false) {
            $bf = new BucketFile($response);
            $bf->set("sourceSha1",$this->fileHash);
            
            $uDeferred->resolve($bf);
            
            //return new BucketFile($response);
        }else {
            $uDeferred->reject("");
        }
        $this->_trace("funcEnd",__METHOD__);
        
        return $uDeferred->promise();
    }

    /**
     * Returns a promise, but uses many internal promises
     * @param bool $timeout 
     * @return false|BucketFile 
     */
    protected function uploadMultiPartAsync($timeout=true,$multiPartSize=null) {
        
        if(is_null($multiPartSize)) {
            $multiPartSize = $this->multiPartSize;
        }
        $uDeferred = new \React\Promise\Deferred();
        $this->_trace("funcStart",__METHOD__);
        $params = [
            'bucketId' => $this->bucketId,
            'fileName' => $this->fileName,
            'contentType' => $this->contentType,
            'fileInfo' => [
                "large_file_sha1" => $this->fileHash,
                "src_last_modified_millis"=> (string) $this->modifiedTime
            ]
        ];
        if(!empty($meta)) {
            $params["fileInfo"] = $meta;
        }
        
        //This call we do synchronously
        $request = $this->client->request("post","/b2_start_large_file",$params);
        if(($response = $request->jsonParseSend()) !== false && isset($response["fileId"])) {
            $newFileId = $response['fileId'];
        } else {
            $uDeferred->reject($response);
            unset($request);
            return $uDeferred->promise();
        }
        unset($response);
        unset($request);
        

        $parts = [];
        $partsTotal = ceil($this->fileSize/$multiPartSize);
        $pendingParts = [];
        $results = [];
        for($i=0;$i<$partsTotal;$i++) {
            $partNumber=$i+1;
            $pendingParts[$partNumber] = null;
            $results[$partNumber] = null;
        }
        $partsDeferred = new Deferred();
        for($i=0;$i<$partsTotal;$i++) {
            //dhGlobal::outLine("part $i");
            $partNumber=$i+1;
            $sizeSent = $i * $multiPartSize;
            $remainingSize = $this->fileSize - $sizeSent;
            $size = $remainingSize > $multiPartSize ? $multiPartSize : $remainingSize;

            $partInfo = dhGlobal::getFileHashAndSize($this->content,$sizeSent,$size);
            $partSize = $partInfo["size"];
            $partHash = $partInfo["hash"];
            $parts[] = $partHash;
            $this->partPieces[$partNumber] = dhGlobal::getPartOfFile($this->content,$sizeSent,$size);
            
            $this->sendPart($newFileId,$partNumber,$partSize,$partHash,$timeout,$uDeferred,$pendingParts,$partsDeferred,$sizeSent,$size,0);
            
        }
        
        $this->content=null;
        

        $partsDeferred->promise()->then(function() use($parts,$newFileId,&$uDeferred,$pendingParts) {
            
            $this->_trace("upload - all parts done");
            //dhGlobal::outLine("parts all done");
            $request = $this->client->request("post","/b2_finish_large_file",["fileId"=>$newFileId,"partSha1Array"=>$parts]);
            $this->_trace("funcEnd",__METHOD__);
            if(($response = $request->jsonParseSend()) !== false) {
                $this->content=null;
                $bf = new BucketFile($response);
                $bf->set("sourceSha1",$this->fileHash);
                $uDeferred->resolve($bf);
                //return new BucketFile($response);
            }else {
                $this->content=null;
                $uDeferred->reject($pendingParts);
            }
            unset($response);
            unset($request);
        });

        //Resolves with BucketFile
        //Rejects with array of results from each of the part uploads
        //May refactor this down into a special thrown Exception/wrapper class
        return $uDeferred->promise();
    }
    private function sendPart($newFileId,$partNumber,$partSize,$partHash,$timeout,&$uDeferred,&$pendingParts,&$partsDeferred,$sizeSent,$size,$tries=0) {
        
        $tries++;
        $partUrlRequest = $this->client->request("post","/b2_get_upload_part_url",["fileId"=>$newFileId]);
        $partUrlRequest->async(true);
        //dhGlobal::outLine("($partNumber) sending to ",$partUrlRequest->getUrl());
        $partUrlRequest->sendAsync(function($response) use(&$partsDeferred,$partUrlRequest,&$uDeferred,&$pendingParts,$timeout,$partNumber,$partSize,$partHash,$sizeSent,$size) {
            
            $this->handlePartUrlRequest($response,$partsDeferred,$partUrlRequest,$uDeferred,$pendingParts,$timeout,$partNumber,$partSize,$partHash,$sizeSent,$size);

        },function($e) use (&$partsDeferred,$partUrlRequest,&$uDeferred,&$pendingParts,$partNumber,$tries,$timeout,$partSize,$partHash,$sizeSent,$size,$newFileId) {
            if($tries<3) {
                $this->sendPart($newFileId,$partNumber,$partSize,$partHash,$timeout,$uDeferred,$pendingParts,$partsDeferred,$sizeSent,$size,$tries=0);
            } else {
                $this->_trace("upload - some kind of general error with $partNumber...",$partUrlRequest->getMethod()." Request to ".$partUrlRequest->getUrl()." failed with exception ".$e->getCode()." - ".$e->getMessage());
                $pendingParts[$partNumber] = $partUrlRequest->getMethod()." Request to ".$partUrlRequest->getUrl()." failed with exception ".$e->getCode()." - ".$e->getMessage();
                if($this->isPartArrayComplete($pendingParts)) {
                    $partsDeferred->resolve(true);
                }
            }
        });
    }
    private function handlePartUrlRequest($response,&$partsDeferred,$partUrlRequest,&$uDeferred,&$pendingParts,$timeout,$partNumber,$partSize,$partHash,$sizeSent,$size) {
        $body = $response->body(true);
        
        if(is_array($body)) {
            $this->_trace("upload - part $partNumber prepped");
            //dhGlobal::outLine("got a response with $partNumber, proceeding to upload");
            $uploadRequest = $this->client->request("post",$body["uploadUrl"])
                ->authToken($body["authorizationToken"])
                ->header("X-Bz-Part-Number",$partNumber)
                ->header("Content-Length",$partSize)
                ->header("X-Bz-Content-Sha1",$partHash)
                ->rawBody($this->partPieces[$partNumber]);
            $uploadRequest->async(true);
            //dhGlobal::outLine("($partNumber) sending to ",$uploadRequest->getUrl());
            $uploadRequest->sendAsync(function($response) use (&$partsDeferred,&$pendingParts,$partNumber,&$uploadRequest) {
                
                $this->_trace("upload - part $partNumber done");
                //dhGlobal::outLine("part $partNumber is complete");
                $pendingParts[$partNumber] = true;
                unset($this->partPieces[$partNumber]);
                $uploadRequest=null;
                if($this->isPartArrayComplete($pendingParts)) {
                    $this->partPieces = null;
                    
                    $partsDeferred->resolve(true);
                }
                
            },function($e) use (&$partsDeferred,$uploadRequest,&$pendingParts,$partNumber,&$uploadRequest) {
                $this->_trace("upload - part $partNumber failed");
                //dhGlobal::outLine("part $partNumber failed");
                $pendingParts[$partNumber] = $uploadRequest->getMethod()." Request to ".$uploadRequest->getUrl()." failed with exception ".$e->getCode()." - ".$e->getMessage();
                unset($this->partPieces[$partNumber]);
                $uploadRequest=null;
                if($this->isPartArrayComplete($pendingParts)) {
                    
                    $partsDeferred->resolve(true);
                }
            },$timeout);
        } else {
            $this->_trace("upload - part $partNumber errored");
            //dhGlobal::outLine("got an err response with $partNumber");
            $pendingParts[$partNumber] = $response->body();
            $this->partPieces = [];
            if($this->isPartArrayComplete($pendingParts)) {
                
                $partsDeferred->resolve(true);
            }
        }
        
    }

    private function isPartArrayComplete($partArray) {
        foreach($partArray as $pn=>$re) {
            if(is_null($re)) {
                return false;
            }
        }
        return true;
    }

    protected function handleMultiResponse() {

    }

    /**
	 * Get the value of uploadNormalLimit
     * @return  mixed
     */
    public function getUploadNormalLimit() {
        return $this->uploadNormalLimit;
    }

    /**
     * Set the value of uploadNormalLimit
     * @param   mixed  $uploadNormalLimit  
     * @return  self
	 */
    public function setUploadNormalLimit($uploadNormalLimit) {
        $this->uploadNormalLimit = $uploadNormalLimit;
        return $this;
    }

    /**
	 * Get the value of multiPartSize
     * @return  mixed
     */
    public function getMultiPartSize() {
        return $this->multiPartSize;
    }

    /**
     * Set the value of multiPartSize
     * @param   mixed  $multiPartSize  
     * @return  self
	 */
    public function setMultiPartSize($multiPartSize) {
        $this->multiPartSize = $multiPartSize;
        return $this;
    }
}