<?php
namespace boru\backblaze;

use boru\backblaze\parts\traits\TraitLog;
use boru\backblaze\parts\UploadPart;
use \boru\backblaze\Client;
use boru\backblaze\exceptions\ClientException;
use \boru\backblaze\parts\BucketFile;
use boru\dhutils\dhGlobal;
use boru\dhutils\filesys\File;
use Exception;
use InvalidArgumentException;
use React\EventLoop\Loop;
use React\Promise\Deferred;
use React\Promise\PromiseInterface;
use React\Stream\ReadableResourceStream;
use UnexpectedValueException;

class UploadBetter {
    private function getMemUsed() {
        return dhGlobal::formatBytes(memory_get_usage());
    }
    private $client;
    
    protected $debug_name = "backblaze";
    protected $debug_prefix = "BBUPLOAD";
    public $debugDoTrace = false;

    //settings
    private $uploadNormalLimit = 10000000; //30mb
    private $multiPartSize = 5000000; //5mb
    private $numParts = 0; //if set, will split the file into this many parts.
    private $timeout = 300;
    private $bucketId;
    private $prefix;

    private $retryCounts = [];
    private $maxRetries = 5;
    private $retryDelay = 3;
    private $retryBackoffPercent = 1.5;
    
    //fileDetails
    /** @var File */
    private $file;
    private $fileName;
    private $modifiedTime;
    private $modifiedTimeMillis;
    private $fileSize;
    private $fileHash;
    private $contentType;
    /** @var ReadableResourceStream|string */
    private $stream;
    private $rawStream;
    private $meta;

    private $largeFileId;

    private $finalized = false;

    /** @var Deferred */
    private $uploadDeferred;
    /** @var Deferred */
    private $multipartDeferred;
    /** @var UploadPart[] */
    private $fileParts = [];

    public function __construct($clientOrOptions) {
        if(is_object($clientOrOptions) && $clientOrOptions instanceof Client) {
            $this->client = $clientOrOptions;
        } elseif(is_array($clientOrOptions)) {
            $this->client = Client::fromAuthArray($clientOrOptions);
        }
        if(is_null($this->client)) {
            throw new \Exception(__CLASS__." - failed constructor, client or auth array invalid");
        }
        $this->debugDoTrace = $this->client->debugDoTrace;
    }
    private function cleanupOnDone() {
        if($this->stream instanceof ReadableResourceStream) {
            $this->stream->close();
        }
        if(!is_null($this->rawStream) && is_resource($this->rawStream)) {
            fclose($this->rawStream);
        }
        $this->file = null;
        $this->stream = null;
        $this->rawStream = null;
        gc_collect_cycles();
    }

    public function handleUploadComplete($bucketFile) {
        $this->finalized = true;
        $this->cleanupOnDone();
        $bucketFile->set("sourceSha1",$this->fileHash);
        $this->_trace("resolved",__METHOD__,__LINE__);

        if(!is_null($this->multipartDeferred)) {
            $this->multipartDeferred->reject('finalized');
        }
        if($bucketFile->sha1() == $this->fileHash && !empty($bucketFile->fileId())) {
            $this->uploadDeferred->resolve($bucketFile);
        } else {
            $this->uploadDeferred->reject("sha1 mismatch");
        }
    }
    public function handleUploadFailed($reason) {
        $this->finalized = true;
        $this->cleanupOnDone();
        $this->uploadDeferred->reject($reason);
        if(!is_null($this->multipartDeferred)) {
            $this->multipartDeferred->reject('finalized');
        }
    }

    public function upload($options=[]) {
        $this->_trace("funcStart",__METHOD__,__LINE__);
        $this->uploadDeferred       = new Deferred();
        $this->bucketId             = $this->getBucketIdFromOptions($options);

        $this->uploadNormalLimit    = dhGlobal::getVal($options, "uploadNormalLimit",   $this->uploadNormalLimit);
        $this->multiPartSize        = dhGlobal::getVal($options, "multiPartSize",       $this->multiPartSize); 
        $this->numParts             = dhGlobal::getVal($options, "numParts",            $this->numParts); 
        $this->timeout              = dhGlobal::getVal($options, "timeout",             true);
        $this->prefix               = dhGlobal::getDot($options, "prefix",              "");

        $this->file                 = dhGlobal::getVal($options, "file",                null);
        $this->fileName             = dhGlobal::getVal($options, "fileName",            false);
        $this->modifiedTimeMillies  = dhGlobal::getVal($options, "modifiedTimeMillis",  false);
        $this->modifiedTime         = dhGlobal::getVal($options, "modifiedTime",        false);
        $this->contentType          = dhGlobal::getVal($options, "contentType",         "b2/x-auto");
        $this->content              = dhGlobal::getVal($options, "content",             null);


        if(($this->stream = $this->prepareUpload($options)) !== false && !is_null($this->stream)) {
            if($this->fileSize<=$this->uploadNormalLimit) {
                //uploadNormal;
                $this->_trace("fileSize = ",$this->fileSize,"..","using uploadNormal.");
                $this->uploadNormal();
            } else {
                //uploadMulti;
                $this->_trace("fileSize = ",$this->fileSize,"..","using uploadMulti.");
                $this->uploadMulti();
            }
        } else {
            $this->handleUploadFailed("poop");
        }
        $this->_trace("funcEnd",__METHOD__,__LINE__);
        return $this->uploadDeferred->promise();
    }
    private function uploadNormal() {
        $this->_trace("funcStart",__METHOD__,__LINE__);
        $this->getUploadUrlAuth()->then(function($authUrlResponse) {
            $request = $this->createNormalUploadRequest($authUrlResponse);
            $request->sendAsync(function($response) use ($request) {
                if(($json = $response->body(true)) && is_array($json)) {
                    $this->handleUploadComplete(new BucketFile($json));
                } else {
                    $this->handleNormalUploadError($response,$request);
                }
                
            }, function($e) use ($request) {
                $this->_trace("rejected",__METHOD__,__LINE__);
                $this->handleUploadFailed($this->makeRejectMessageFromRequest($e,$request));
            },$this->timeout);
        });
        $this->_trace("funcEnd",__METHOD__,__LINE__);
    }
    
    private function uploadMulti() {
        $this->_trace("funcStart",__METHOD__,__LINE__);
        if(($this->largeFileId = $this->startLargeFile()) === false) {
            $this->handleUploadFailed("Unable to get largeFileId");
            return;
        }
        $this->fileParts = $this->createMultiParts();
        if(empty($this->fileParts)) {
            $this->handleUploadFailed("empty parts");
            return;
        }
        $this->uploadMultiParts()->then(function() {
            $this->finishLargeFile();
        },function($e) {
            $this->_trace("errorHandled",__METHOD__,__LINE__);
        });
        
    }
    private function finishLargeFile() {
        $parts = [];
        foreach($this->fileParts as $i=>$uploadPart) {
            $parts[] = $uploadPart->hash();
            $this->fileParts[$i] = null;
            unset($this->fileParts[$i]);
        }
        $request = $this->client->request("post","/b2_finish_large_file",["fileId"=>$this->largeFileId,"partSha1Array"=>$parts]);
        if(($response = $request->jsonParseSend()) !== false) {
            $this->handleUploadComplete(new BucketFile($response));
        } else {
            $this->handleUploadFailed("failed to finish large file");
        }
    }
    private function uploadMultiParts() {
        $this->_trace("funcStart",__METHOD__,__LINE__);
        $this->multipartDeferred = new Deferred();
        foreach($this->fileParts as $uploadPart) {
            $this->sendMultiPart($uploadPart);
        }
        Loop::addPeriodicTimer(0.1,function($timer) {
            foreach($this->fileParts as $i=>$uploadPart) {
                if(!$uploadPart->done()) {
                    return;
                }
                $uploadPart->set("content","");
            }
            Loop::cancelTimer($timer);
            $this->multipartDeferred->resolve(true);
            $this->_trace("funcEnd",__METHOD__,__LINE__);
        });
        return $this->multipartDeferred->promise();
    }
    private function sendMultiPart(UploadPart $uploadPart) {
        $this->_trace("funcStart",__METHOD__,__LINE__);
        $uploadPart->incAttempts();
        $uploadPart->started(true);
        $uploadPart->done(false);
        $this->getUploadPartUrl(function($uploadInfo) use ($uploadPart) {
            $this->_trace("funcEnd",__METHOD__,__LINE__);
            $this->uploadMultiPartPart($uploadInfo,$uploadPart);
        },function($error,$request) use($uploadPart) {
            $this->_trace("funcEnd",__METHOD__,__LINE__);
            if(!$this->finalized) {
                if($uploadPart->attempts()<3) {
                    $this->sendMultiPart($uploadPart);
                } else {
                    //failed!
                    $this->handleUploadFailed("failed 3 times on sendMultiPart(".$uploadPart->number().")");
                }
            }
        });
    }
    private function uploadMultiPartPart($uploadInfo,UploadPart $uploadPart) {
        $this->_trace("funcStart",__METHOD__,__LINE__);
        $request = $this->client->request("post",$uploadInfo["uploadUrl"])
            ->authToken($uploadInfo["authorizationToken"])
            ->header("X-Bz-Part-Number",$uploadPart->number())
            ->header("Content-Length",$uploadPart->size())
            ->header("X-Bz-Content-Sha1",$uploadPart->hash());
        $request->rawBody($uploadPart->content());
        $request->async(true);
        $request->sendAsync(function($response) use ($uploadPart) {
            $uploadPart->done(true);
            $uploadPart->started(false);
            $uploadPart->content(" ");
            $this->_trace("funcEnd",__METHOD__,__LINE__);
        },function($error) use ($uploadPart) {
            $this->_trace("funcEnd",__METHOD__,__LINE__);
            if(!$this->finalized) {
                if($uploadPart->attempts()<3) {
                    $this->sendMultiPart($uploadPart);
                } else {
                    //failed!
                    $this->handleUploadFailed("failed 3 times on uploadMultiPartPart(".$uploadPart->number().")");
                }
            }
        },$this->timeout);
    }

    private function createMultiParts() {
        $this->_trace("funcStart",__METHOD__,__LINE__);
        $totalParts = ceil($this->fileSize/$this->multiPartSize);
        $sizeUsed=0;
        $i=0;
        $fileParts = [];
        while($this->fileSize > $sizeUsed) {
            $partNumber = $i+1;
            $sizeUsed = $i * $this->multiPartSize;
            $remainingSize = $this->fileSize - $sizeUsed;
            $partSize = $remainingSize > $this->multiPartSize ? $this->multiPartSize : $remainingSize;
            $partInfo = dhGlobal::getfileHashAndSize($this->rawStream,$sizeUsed,$partSize);
            $fileParts[$i] = new UploadPart($partNumber,$partSize,$partInfo["hash"],$sizeUsed);
            $fileParts[$i]->content(stream_get_contents($this->rawStream,$partSize,$sizeUsed));
            $i++;
            $sizeUsed+=$partSize;
        }
        $this->_trace("funcEnd",__METHOD__,__LINE__);
        return $fileParts;
    }


    private function handleNormalUploadError($response,$request) {
        $this->_trace("funcStart",__METHOD__,__LINE__);
        if(is_object($response)) {
            if($response->getStatusCode() == 401 && $response->getReasonPhrase() == "unauthorized") {
                $this->_trace("rejected",__METHOD__,__LINE__);
                $this->handleUploadFailed("401-unauthorized returned from BackBlaze. Account credentials are not valid to perform this action");
            } elseif($response->getStatusCode() == 401 || $response->getStatusCode() == 400) {
                //retry with backoff
                if(!$this->retriesWithBackoff($this->retryCounts["normal"],function() { $this->uploadNormal(); })) {
                    $this->_trace("rejected",__METHOD__,__LINE__);
                    $this->handleUploadFailed("upload re-try limit (".$this->maxRetries.") hit");
                }
                return;
            } else {
                //request failed..
                $this->_trace("rejected",__METHOD__,__LINE__);
                $this->handleUploadFailed($this->makeRejectMessageFromRequest($response,$request));
            }
        } else {
            $this->_trace("rejected",__METHOD__,__LINE__);
            $this->handleUploadFailed($this->makeRejectMessageFromRequest($response,$request));
        }
    }
    private function retriesWithBackoff(&$retries=0,$callable) {
        $this->_trace("funcStart",__METHOD__,__LINE__);
        if($retries>=$this->maxRetries) {
            return false;   
        } else {
            $retries++;
            $delay = $this->retryDelay * (pow($this->retryBackoffPercent,$retries));
            usleep($delay);
            $callable();
            return true;
        }
    }

    private function prepareUpload() {
        $this->_trace("funcStart",__METHOD__,__LINE__);
        $stream = false;
        $this->prefix = dhGlobal::trimString("/",$this->prefix,dhGlobal::TRIM_BOTH);
        if(!is_null($this->file) && $this->file instanceof File) {
            if(is_null($this->fileName)) {
                $this->fileName           = dhGlobal::ltrimString("/",$this->prefix.$this->file->path());
            }
            $this->modifiedTimeMillis = $this->file->mtime("millis");
            $this->fileSize           = $this->file->size();
            $this->fileHash           = $this->file->sha1();
            $this->contentType        = $this->file->mimeType();
            $this->rawStream          = fopen($this->file->path(),"r");
            $stream = new ReadableResourceStream($this->rawStream);
        } elseif(is_resource($this->file) && ($fstat = fstat($this->file)) !== false) {
            $info = dhGlobal::getFileHashAndSize($this->file);
            $this->fileSize = $info["size"];
            $this->fileHash = $info["hash"];
            $this->modifiedTime = $fstat["mtime"];
            if(($mimeType = mime_content_type($this->file)) !== false) {
                $this->contentType = $mimeType;
            }
            $this->rawStream  = &$this->file;
            $stream = new ReadableResourceStream($this->file);
        } else {
            $stream = &$this->content;
            $fileInfo = dhGlobal::getFileHashAndSize($stream);
            $this->fileSize = $fileInfo["size"];
            $this->fileHash = $fileInfo["hash"];
        }

        if(!is_null($this->numParts) && $this->numParts>0) {
            $maxParts = max(5,floor(($this->fileSize/$this->uploadNormalLimit) * $this->numParts));
            $this->multiPartSize = min(max(5000000,floor($this->fileSize/$maxParts)),$this->uploadNormalLimit);
        }

        $this->modifiedTimeOverride();
        $this->_trace("funcEnd",__METHOD__,__LINE__);
        return $stream;
    }

    public function startLargeFile() {
        $this->_trace("funcStart",__METHOD__,__LINE__);
        $params = [
            'bucketId' => $this->bucketId,
            'fileName' => $this->fileName,
            'contentType' => $this->contentType,
            'fileInfo' => [
                "large_file_sha1" => $this->fileHash,
                "src_last_modified_millis"=> (string) $this->modifiedTimeMillis
            ]
        ];
        if(!empty($this->meta)) {
            foreach($this->meta as $k=>$v) {
                $params["fileInfo"][$k] = $v;
            }
        }
        $request = $this->client->request("post","/b2_start_large_file",$params);
        if(($response = $request->jsonParseSend()) !== false && isset($response["fileId"])) {
            unset($request);
            return $response['fileId'];
        } else {
            unset($request);
            unset($response);
            return false;
        }
    }

    private function getUploadPartUrl($onSuccess,$onError) {
        $partUrlRequest = $this->client->request("post","/b2_get_upload_part_url",["fileId"=>$this->largeFileId]);
        $partUrlRequest->async(true);
        $partUrlRequest->sendAsync(function($response) use($onSuccess,$onError,$partUrlRequest) {
            $body = $response->body(true);
            if(is_array($body)) {
                $onSuccess($body);
            } else {
                $onError($response->body,$partUrlRequest);
            }
        },function($e) use ($onError,$partUrlRequest) {
            $onError($e,$partUrlRequest);
        });
    }

    /**
     * 
     * @param string|Bucket $bucket 
     * @param bool $cache 
     * @return \React\Promise\Promise 
     * @throws ClientException 
     * @throws ClientException 
     */
    private function getUploadUrlAuth($cache=false) {
        $this->_trace("funcStart",__METHOD__,__LINE__);
        $deferred = new Deferred();
        if($cache && ($response = $this->client->getUploadAuth()) !== false) {
            $deferred->resolve($response);
        } else {
            $params = ["bucketId"=>$this->bucketId];
            $request = $this->client->request("post","/b2_get_upload_url",$params);
            $request->sendAsync(function($response) use ($deferred,$cache) {
                $json = $response->body(true);
                if(is_array($json)) {
                    $this->client->accountSet("uploadAuth",$json,$cache);
                }
                $this->_trace("resolved",__METHOD__,__LINE__);
                $deferred->resolve($json);
            },function($e) use ($deferred,$cache) {
                $this->_trace("rejected",__METHOD__,__LINE__);
                $deferred->reject($e);
            });
        }
        $this->_trace("funcEnd",__METHOD__,__LINE__);
        return $deferred->promise();
    }

    private function getBucketIdFromOptions($options=[],$exception=true) {
        $this->_trace("funcStart",__METHOD__,__LINE__);
        $bucketId = false;
        if(($bucketId = dhGlobal::getDot($options,"bucketId",false))!==false) {

        } elseif(($bucket = $this->client->buckets()->fromOptions($options)) !== false) {
            $bucketId = $bucket->get("bucketId");
        } else {
            if($exception) {
                throw new ClientException(__METHOD__,"No bucket option provided (bucket, bucketId, bucketName)");
            }
        }
        $this->_trace("funcEnd",__METHOD__,__LINE__);
        return $bucketId;
    }
    private function modifiedTimeOverride() {
        $this->_trace("funcStart",__METHOD__,__LINE__);
        if(!is_null($this->modifiedTime) && is_null($this->modifiedTimeMillis)) {
            if(is_object($this->modifiedTime)) {
                $this->modifiedTimeMillis = $this->modifiedTime->format("U")*1000;
            } elseif(is_numeric($this->modifiedTime)) {
                $this->modifiedTimeMillis = $this->modifiedTime*1000;
            } else {
                try {
                    $mt = new \DateTime($this->modifiedTime);
                    $this->modifiedTimeMillis = $mt->format("U")*1000;
                } catch (\Exception $e) {

                }
            }
        }
        if(is_null($this->modifiedTimeMillis)) {
            $this->modifiedTimeMillis = floor(microtime(true) * 1000);
        }
        $this->_trace("funcEnd",__METHOD__,__LINE__);
    }
    private function makeRejectMessageFromRequest($response,$request) {
        $this->_trace("funcStart",__METHOD__,__LINE__);
        if(is_object($response) && method_exists($response,"getMessage")) {
            return $request->getMethod()." Request to ".$request->getUrl()." failed with exception ".$response->getCode()." - ".$response->getMessage();
        } else {
            if(is_array($response)) {
                $response = json_encode($response);
            }
            return $request->getMethod()." Request to ".$request->getUrl()." failed with non-object: ".$response;
        }
    }

    private function createNormalUploadRequest($authUrlResponse) {
        $this->_trace("funcStart",__METHOD__,__LINE__);
        $uploadUrl = $authUrlResponse["uploadUrl"];
        $authToken = $authUrlResponse["authorizationToken"];
        $request = $this->client->request("post",$uploadUrl)
            ->authToken($authToken)
            ->header("Content-Type",$this->contentType)
            ->header("Content-Length",$this->fileSize)
            ->header("X-Bz-File-Name",$this->fileName)
            ->header("X-Bz-Content-Sha1",$this->fileHash)
            ->header("X-Bz-Info-src_last_modified_millis",$this->modifiedTimeMillis)
            ->rawBody($this->stream)
            ->async(true);
        if(!empty($this->meta)) {
            foreach($this->meta as $k=>$v) {
                $request->header("X-Bz-Info-".$k,$v);
            }
        }
        //print_r($request);
        $this->_trace("funcEnd",__METHOD__,__LINE__);
        return $request;
    }

    private $lastMemUsed = 0;
    protected function _trace(...$args) {
        $diff = memory_get_usage() - $this->lastMemUsed;
        $this->lastMemUsed = memory_get_usage();
        if($this->debugDoTrace) {
            array_unshift($args,"[MemUsed:".dhGlobal::formatBytes(memory_get_usage())." / Diff:".dhGlobal::formatBytes($diff)."]  ");
            array_unshift($args,$this->debug_prefix." - ");
            array_unshift($args,$this->debug_name."-t");
            return call_user_func_array([$this,"log"],$args);
        }
        return;
    }
    protected function _debug(...$args) {
        array_unshift($args,$this->debug_prefix." - ");
        array_unshift($args,$this->debug_name."-d");
        return call_user_func_array([$this,"log"],$args);
    }
    protected function _info(...$args) {
        array_unshift($args,$this->debug_prefix." - ");
        array_unshift($args,$this->debug_name."-i");
        return call_user_func_array([$this,"log"],$args);
    }
    protected function _warn(...$args) {
        array_unshift($args,$this->debug_prefix." - ");
        array_unshift($args,$this->debug_name."-w");
        return call_user_func_array([$this,"log"],$args);
    }
    protected function _error(...$args) {
        array_unshift($args,$this->debug_prefix." - ");
        array_unshift($args,$this->debug_name."-e");
        return call_user_func_array([$this,"log"],$args);
    }
    public function log(...$args) {
        return dhGlobal::log(...$args);
    }
}