<?php
namespace boru\backblaze;

use boru\backblaze\async\AsyncWorker;
use boru\backblaze\lib\Http;
use boru\backblaze\lib\Request;
use boru\backblaze\parts\Bucket;
use boru\backblaze\parts\traits\TraitLog;
use boru\dhutils\async\Queue;
use boru\dhutils\async\Work;
use boru\dhutils\dhGlobal;

class Client {

    const ERR_UPLOAD_SHA1_MISMATCH = "sha1-mismatch";
    const ERR_BLANK_RESPONSE = "blank-response";
    const ERR_NOT_BUCKETFILE = "not-bucketfile";

    use TraitLog;
    public $apiUrl = 'https://api.backblazeb2.com';
    public $apiVersion = "/b2api/v2";

    protected $client;

    protected $cacheFile;

    protected $uploadTries = 0;
    protected $uploadDelay = 1; //seconds
    protected $uploadBackoff = 1.2; //+20%

    protected $_bucketCacheLastList = 0;
    protected $_bucketCache = [];

    protected $debug_name = "backblaze";
    protected $debug_prefix = "BACKBLAZE";
    public $debugDoTrace = false;

    /** @var \boru\backblaze\lib\Http */
    protected $http;

    /** @var \boru\backblaze\Buckets */
    protected $bucketOperations;

    /** @var \boru\backblaze\Files */
    protected $fileOperations;

    /** @var \boru\backblaze\Upload */
    protected $uploadOperations;

    /** @var \boru\dhutils\async\Queue */
    private $queue;

    private $authArray;

    public function __construct($keyId,$applicationKey,$options=[]) {
        $this->setupDebug();
        $this->authArray["keyId"] = $keyId;
        $this->authArray["applicationKey"] = $applicationKey;
        $this->authArray["options"] = $options;
        $this->http = new Http($keyId,$applicationKey,$options);
        $account = $this->http->getAccount();
        if(!$account->checkAuthorized()) {
            throw new \Exception("Authorization Failure");
        }
    }
    public function dropHttp() {
        $this->http = null;
    }
    public function resetHttp() {
        $this->http = new Http($this->authArray["keyId"],$this->authArray["applicationKey"],$this->authArray["options"]);
    }

    /** @return Files */
    public function files() {
        if(is_null($this->fileOperations)) {
            $this->fileOperations = new Files($this);
        }
        return $this->fileOperations;
    }
    public function upload() {
        if(is_null($this->uploadOperations)) {
            $this->uploadOperations = new Upload($this);
        }
        return $this->uploadOperations;
    }
    public function buckets() {
        if(is_null($this->bucketOperations)) {
            $this->bucketOperations = new Buckets($this);
        }
        return $this->bucketOperations;
    }

    public function getAccount($item=null,$default=false) {
        $this->_trace("funcStart",__METHOD__);
        if(is_null($this->accountInfo)) {
            $this->_trace("funcEnd",__METHOD__);
            return $default;
        }
        if(is_null($item)) {
            $this->_trace("funcEnd",__METHOD__);
            return $this->accountInfo;
        } elseif(!isset($this->accountInfo[$item])) {
            $this->_trace("funcEnd",__METHOD__);
            return $default;
        }
        $this->_trace("funcEnd",__METHOD__);
        return $this->accountInfo[$item];
    }

    /**
     * 
     * @param mixed $method 
     * @param mixed $url 
     * @param mixed $data 
     * @param mixed $auth 
     * @param mixed $onSuccess 
     * @param mixed $onError 
     * @return Request 
     */
    public function request($method=null,$url=null,$data=null,$auth=true,$onSuccess=null,$onError=null) {
        if(is_null($this->http)) {
            $this->resetHttp();
        }
        return $this->http->request($method,$url,$data,$auth,$onSuccess,$onError);
    }

    public function accountSet($dotKey,$value=null,$writeCache=false) {
        if(is_null($this->http)) {
            $this->resetHttp();
        }
        $this->http->accountSet($dotKey,$value,$writeCache);
        return $this;
    }
    public function getAccountId() {
        if(is_null($this->http)) {
            $this->resetHttp();
        }
        return $this->http->getAccount()->get("accountId",false);
    }
    public function getDownloadUrl() {
        if(is_null($this->http)) {
            $this->resetHttp();
        }
        return $this->http->getAccount()->get("downloadUrl",false);
    }
    public function getUploadAuth() {
        if(is_null($this->http)) {
            $this->resetHttp();
        }
        return $this->http->getAccount()->get("uploadAuth",false);
    }
    /**
     * Get the value of keyId
     */ 
    public function getKeyId() {
        if(is_null($this->http)) {
            $this->resetHttp();
        }
        return $this->http->getAccount()->getKeyId();
    }

    /**
     * Set the value of keyId
     *
     * @return  self
     */ 
    public function setKeyId($keyId) {
        if(is_null($this->http)) {
            $this->resetHttp();
        }
        $this->http->getAccount()->setKeyId($keyId);
        return $this;
    }

    /**
     * Get the value of applicationKey
     */ 
    public function getApplicationKey() {
        if(is_null($this->http)) {
            $this->resetHttp();
        }
        return $this->http->getAccount()->getApplicationKey();
    }

    /**
     * Set the value of applicationKey
     *
     * @return  self
     */ 
    public function setApplicationKey($applicationKey) {
        if(is_null($this->http)) {
            $this->resetHttp();
        }
        $this->http->getAccount()->setApplicationKey($applicationKey);
        return $this;
    }


    /**
	 * Get the value of queue
     * @return  mixed
     */
    public function getQueue() {
        return $this->queue;
    }

    /**
     * Set the value of queue
     * @param   mixed  $queue  
     * @return  self
	 */
    public function setQueue($queue) {
        $this->queue = $queue;
        return $this;
    }
    private function pathFromFileOrPath($fileOrPath) {
        if(is_object($fileOrPath) && method_exists($fileOrPath,"path")) {
            $filePath = $fileOrPath->path();
        } elseif(is_object($fileOrPath) || is_array($fileOrPath)) {
            return false;
        } else {
            $filePath = $fileOrPath;
        }
        return $filePath;
    }

    public function queueVerifyById($sourceFileOrPath,$fileId,$extraArgs=[],$bootstrap=null,$run=true) {
        if(($filePath = $this->pathFromFileOrPath($sourceFileOrPath)) === false) {
            return false;
        }
        $args = [
            "sourceFile"=>$filePath,
            "fileId"=>$fileId
        ];
        if(!is_null($extraArgs) && !empty($extraArgs)) {
            foreach($extraArgs as $k=>$v) {
                $args[$k] = $v;
            }
        }
        return $this->asyncWorker("verifyUpload",$args,$bootstrap,$run);
    }
    public function queueUpload($bucketOrId,$fileOrPath,$destFilePath,$extraArgs=[],$bootstrap=null,$run=true) {
        if(is_object($bucketOrId) && $bucketOrId instanceof Bucket) {
            $bucketId = $bucketOrId->get("bucketId");
        } elseif(is_object($bucketOrId) || is_array($bucketOrId)) {
            return false;
        } else {
            $bucketId = $bucketOrId;
        }
        if(($filePath = $this->pathFromFileOrPath($fileOrPath)) === false) {
            return false;
        }
        
        $args = [
            "bucketId"=>$bucketId,
            "sourceFile"=>$filePath,
            "destFile"=>$destFilePath
        ];
        if(!is_null($extraArgs) && !empty($extraArgs)) {
            foreach($extraArgs as $k=>$v) {
                $args[$k] = $v;
            }
        }
        return $this->asyncWorker("upload",$args,$bootstrap,$run);
    }
    public function asyncWorker($methodName,$args=[],$bootstrap=null,$run=true) {
        $appKey = $this->getApplicationKey();
        $keyId = $this->getKeyId();
        $config = [
            "keyId"=>$keyId,
            "appKey"=>$appKey
        ];
        
        $work = new Work(["\\".AsyncWorker::getClassCallable(),$methodName],["args"=>[$config,$args],"asJson"=>false]);
        if($run) {
            dhGlobal::asyncWork($work,$bootstrap);
        }
        return $work;
    }

    public static function fromAuthArray($array,$throwOnError=true) {
        if(($keyId = dhGlobal::getVal($array,"keyId",false)) === false) {
            if($throwOnError) {
                throw new \Exception("Client::fromAuthArray() -- missing keyId");
            }
            return false;
        }
        if(($applicationKey = dhGlobal::getVal($array,"applicationKey",false)) === false) {
            if(($applicationKey = dhGlobal::getVal($array,"appKey",false)) === false) {
                if($throwOnError) {
                    throw new \Exception("Client::fromAuthArray() -- missing applicationKey");
                }
                return false;
            }
        }
        if(($options = dhGlobal::getVal($array,"options",false)) === false) {
            $options = [];
        }
        if(!is_array($options)) {
            $options = [];
        }
        return new self($keyId,$applicationKey,$options);
    }
}



