<?php
namespace boru\backblaze\lib;

use boru\dhutils\http\Response;
use InvalidArgumentException;
use Exception;
use Throwable;
use UnexpectedValueException;

class Request {
    protected $headers;
    protected $method;
    protected $url;
    protected $onSuccess;
    protected $onError;
    protected $bodyJson;
    protected $bodyForm;
    protected $bodyQuery;
    protected $bodyRaw;
    protected $async = true;

    public function __construct($method=null,$url=null,$data=null,$auth=null,$onSuccess=null,$onError=null) {
        if(!is_null($method)) {
            $this->setMethod($method);
        }
        if(!is_null($url)) {
            $this->setUrl($url);
        }
        if(!is_null($auth)) {
            if(is_array($auth)) {
                $this->authBasic($auth);
            } else {
                $this->authToken($auth);
            }
        }
        if(!is_null($data)) {
            $this->json($data);
        }
        $this->onSuccess = function(\boru\dhutils\http\Response $response) { return $response; };
        if(!is_null($onSuccess)) {
            $this->async = true;
            $this->onSuccess = $onSuccess;
        }
        $this->onError = function($e) { return $e; };
        if(!is_null($onError)) {
            $this->onError = $onError;
        }
    }
    public function async($async=true) {
        $this->async = $async;
        return $this;
    }

    public function rawBody($body) {
        $this->bodyRaw = $body;
        return $this;
    }
    public function json($body=[]) {
        $this->bodyJson = $body;
        return $this;
    }
    public function form($body=[]) {
        $this->bodyForm = $body;
        return $this;
    }
    public function query($body=[]) {
        $this->bodyQuery = $body;
        return $this;
    }
    public function header($key,$value=null) {
        $this->headers[$key]=$value;
        return $this;
    }

    public function send() {
        if(!empty($this->bodyQuery)) {
            if(strpos($this->url,"?") === false) {
                $this->url.="?";
            } else {
                $this->url.="&";
            }
            $this->url .= http_build_query($this->bodyQuery);
        }
        $client = new \boru\dhutils\dhHttp();
        $req = $client->request($this->method,$this->url);
        if(!empty($this->bodyForm)) {
            $req->form($this->bodyForm);
        }
        if(!empty($this->bodyJson)) {
            $req->json($this->bodyJson);
        }
        if(!empty($this->bodyRaw)) {
            $req->body($this->bodyRaw);
        }
        if(!empty($this->headers)) {
            foreach($this->headers as $key=>$value) {
                $req->header($key,$value);
            }
        }
        return $req->send();
    }

    /**
     * 
     * @param callable|null $onSuccess 
     * @param callabel|null $onError 
     * @return \React\Promise\ExtendedPromiseInterface
     * @throws InvalidArgumentException 
     * @throws Exception 
     * @throws Throwable 
     * @throws UnexpectedValueException 
     */
    public function sendAsync($onSuccess=null,$onError=null,$timeout=true) {
        if(is_null($onSuccess)) {
            $onSuccess = $this->onSuccess;
        }
        if(is_null($onError)) {
            $onError = $this->onError;
        }
        if(!empty($this->bodyQuery)) {
            if(strpos($this->url,"?") === false) {
                $this->url.="?";
            } else {
                $this->url.="&";
            }
            $this->url .= http_build_query($this->bodyQuery);
        }
        $options = ["success"=>$onSuccess,"error"=>$onError,"async"=>$this->async,"timeout"=>$timeout];
        $client = \boru\dhutils\dAsyncHttp::client($options);
        if(!empty($this->bodyForm)) {
            $client->form($this->bodyForm);
        }
        if(!empty($this->bodyJson)) {
            $client->json($this->bodyJson);
        }
        if(!empty($this->bodyRaw)) {
            $client->raw($this->bodyRaw);
        }
        if(!empty($this->headers)) {
            foreach($this->headers as $key=>$value) {
                $client->header($key,$value);
            }
        }
        $client->url($this->url);
        return $client->send($this->method);
    }
    public function jsonParseSend($async=false) {
        $response = $this->send();
        if($response instanceof Response) {
            $json = $response->body(true);
            if(is_array($json)) {
                return $json;
            }
        } else {
            if(method_exists($response,"getMessage")) {
                echo $response->getMessage()."\n";
            }
        }
        return false;
    }

    public function authBasic($userPass=[]) {
        $this->headers["Authorization"] = "Basic ".base64_encode(implode(":",$userPass));
        return $this;
    }
    public function authToken($token) {
        $this->headers["Authorization"] = $token;
        return $this;
    }

    /**
     * Get the value of method
     *
     * @return  mixed
     */
    public function getMethod() {
        return $this->method;
    }

    /**
     * Set the value of method
     *
     * @param   mixed  $method  
     * @return  self
     */
    public function setMethod($method) {
        $this->method = $method;
        return $this;
    }

    /**
     * Get the value of url
     *
     * @return  mixed
     */
    public function getUrl() {
        return $this->url;
    }

    /**
     * Set the value of url
     *
     * @param   mixed  $url  
     * @return  self
     */
    public function setUrl($url) {
        $this->url = $url;
        return $this;
    }

    /**
     * Get the value of onSuccess
     *
     * @return  mixed
     */
    public function getOnSuccess() {
        return $this->onSuccess;
    }

    /**
     * Set the value of onSuccess
     *
     * @param   mixed  $onSuccess  
     * @return  self
     */
    public function setOnSuccess($onSuccess) {
        $this->onSuccess = $onSuccess;
        return $this;
    }

    /**
     * Get the value of onError
     *
     * @return  mixed
     */
    public function getOnError() {
        return $this->onError;
    }

    /**
     * Set the value of onError
     *
     * @param   mixed  $onError  
     * @return  self
     */
    public function setOnError($onError) {
        $this->onError = $onError;
        return $this;
    }

    public function toArray() {
        return [
            "url"=>$this->url,
            "method"=>$this->method,
            "headers"=>$this->headers,
            /*"bodyForm"=>$this->bodyForm,
            "bodyJson"=>$this->bodyJson,
            "bodyRaw"=>$this->bodyRaw,*/
            "async"=>$this->async
        ];
    }
}