<?php
namespace boru\backblaze;

use boru\backblaze\browser\File;
use boru\backblaze\browser\Folder;
use boru\backblaze\browser\Listing;
use boru\backblaze\parts\BucketFile;

class Browser {
    /** @var Client */
    private $client;
    private $delimiter = "/";

    /** @var Files */
    private $filesOp;

    private $path = "";
    private $bucketId;

    /** @var Listing[] */
    private $cache = [];

    public function __construct($clientOrOptions,$bucketId) {
        if(is_object($clientOrOptions) && $clientOrOptions instanceof Client) {
            $this->client = $clientOrOptions;
        } elseif(is_array($clientOrOptions)) {
            $this->client = Client::fromAuthArray($clientOrOptions);
        }
        if(is_null($this->client)) {
            throw new \Exception(__CLASS__." - failed constructor, client or auth array invalid");
        }
        $this->filesOp = new Files($this->client);

        $this->bucketId = $bucketId;
    }

    /**
     * 
     * @param mixed $path 
     * @param bool $useCache (true)
     * @param int $maxFileCount (1000)
     * @return Listing 
     */
    public function contents($path=null,$useCache=true,$maxFileCount=1000) {
        $path = $this->path($path);
        $listing = null;
        if(isset($this->cache[$path]) && $useCache) {
            $listing = $this->cache[$path];
        }
        if(!isset($this->cache[$path]) || !$useCache) {
            $options = [
                "bucketId"=>$this->bucketId,
                "maxFileCount"=>$maxFileCount,
                "delimiter"=>$this->delimiter,
                "arrayFormat"=>"name",
                "prefix"=>$this->path($path)
            ];
            $files = $this->filesOp->listFiles($options);
            if(!empty($files)) {
                $listing = Listing::fromBucketFiles($files,$this);
                if($useCache) {
                    $this->cache[$path] = $listing;
                }
            }
        }
        if(is_null($listing)) {
            $listing = new Listing([],[]);
        }
        return $listing;
    }
    public function downloadByFileName($fileName,$headersOnly=false,$async=false) {
        return $this->client->files()->downloadByName($fileName,$headersOnly,$async);
    }
    public function downloadByFileId($fileId,$headersOnly=false,$async=false) {
        return $this->client->files()->downloadById($fileId,$headersOnly,$async);
    }
    public function deleteFile($fileName,$fileId) {
        return $this->client->files()->deleteFile($this->bucketId,$fileName,$fileId);
    }

    /**
     * 
     * @param string $fileName 
     * @param int $limit 
     * @return File[]
     */
    public function getFileVersionsByName($fileName,$limit=100) {
        $versions = $this->client()->files()->versions([
            "bucketId" => $this->bucketId,
            "maxFileCount"=>$limit,
            "delimiter"=>"/",
            "fileName"=>$fileName,
            "prefix"=>$fileName,
            //"maxIterations"=>5,
            //"filesPerRequest"=>1000
        ]);
        if(!empty($versions)) {
            $output = [];
            foreach($versions as $bucketFile) {
                $output[] = static::fileOrFolder($bucketFile,$this);
            }
            return $output;
        }
        return [];
    }

    /**
     * Returns the delimiter-safe current path, optionally changing $this->path before returning it.
     * @param string|null $path if null, returns $this->path, otherwise sets $this->path and then returns it
     * @return string $this->path 
     */
    public function path($path=null) {
        if(is_null($path)) {
            return $this->getPath();
        } else {
            $this->setPath($path);
            return $this->getPath();
        }
    }

    private function pathWithDelimiter($path="") {
        if(empty($path) || $path == $this->delimiter) {
            return "";
        }
        if(substr($path,-(strlen($this->delimiter))) != $this->delimiter) {
            $path.=$this->delimiter;
        }
        return $path;
    }

    /**
	 * Get the value of path
     * @return  mixed
     */
    public function getPath() {
        return $this->pathWithDelimiter($this->path); 
    }

    /**
     * Set the value of path
     * @param   mixed  $path  
     * @return  self
	 */
    public function setPath($path) {
        $this->path = $path;
        return $this;
    }

    /**
	 * Get the value of delimiter
     * @return  mixed
     */
    public function getDelimiter() {
        return $this->delimiter;
    }

    /**
     * Set the value of delimiter
     * @param   mixed  $delimiter  
     * @return  self
	 */
    public function setDelimiter($delimiter) {
        $this->delimiter = $delimiter;
        return $this;
    }

    public function client() {
        return $this->client;
    }
    public function getClient() {
        return $this->client;
    }

    /**
     * 
     * @param BucketFile $bucketFile 
     * @param Browser|null $browser 
     * @return Folder|File 
     */
    public static function fileOrFolder(BucketFile $bucketFile,$browser=null) {
        if($bucketFile->get("action") == "folder") {
            return new Folder($bucketFile);
        } else {
            return new File($bucketFile);
        }
    }

    /**
	 * Get the value of bucketId
     * @return  mixed
     */
    public function getBucketId() {
        return $this->bucketId;
    }
}