<?php
namespace boru\backblaze\async;

use boru\backblaze\Client;
use boru\backblaze\parts\BucketFile;
use boru\backblaze\parts\UploadException;
use boru\dhutils\dhGlobal;
use boru\dhutils\filesys\File;
use React\Promise\Deferred;

class AsyncWorker {
    public static $keyId;
    public static $appKey;
    public static $cacheFile;

    /** @var \boru\backblaze\Client */
    public static $client;

    public static function init($options=[]) {
        if(!is_array($options)) {
            throw new \Exception(__CLASS__."::init requires 1 arg, it should be an array",1);
        }
        static::$keyId = dhGlobal::getVal($options,"keyId",false);
        static::$appKey = dhGlobal::getVal($options,"appKey",false);
        static::$cacheFile = dhGlobal::getVal($options,"cacheFile",".bbcache_".uniqid());
        static::$client = new Client(static::$keyId,static::$appKey,["cacheFile"=>static::$cacheFile,"cacheRemoveOnClose"=>true]);
        return ["success"=>true,"method"=>"init"];
    }
    public static function getTimeoutByFilesize($fileSize) {
        if(!is_null($fileSize)) {
            if($fileSize<500000) { //500k
                $timeOutSeconds = 30;
            } elseif($fileSize<5000000) { //5mb
                $timeOutSeconds = 60;
            } elseif($fileSize<10000000) { //10mb
                $timeOutSeconds = 120;
            } else {
                $timeOutSeconds = 300;
            }
            return $timeOutSeconds;
        }
        return false;
    }
    public static function verifyUpload($config=[],$options=[]) {
        if(!is_array($config) || !is_array($options)) {
            throw new \Exception(__CLASS__."::verifyUpload - requires 2 args, arg1 is the config array, arg2 is the verifyInfo array",1);
        }
        if(($sourceFile = dhGlobal::getval($options,"sourceFile",false)) === false) {
            throw new \Exception(__CLASS__."::upload - missing required option: sourceFile",2);
        }
        if(($destFile = dhGlobal::getval($options,"destFile",false)) === false) {
            if(($fileId = dhGlobal::getVal($options,"fileId",false)) === false) {
                throw new \Exception(__CLASS__."::upload - missing required option: destFile or fileId",2);
            } else {

            }
        }
        if(($file = dhGlobal::fileIfExists($sourceFile)) === false) {
            throw new \Exception(__CLASS__."::upload - sourceFile not a valid file",3);
        }
        $timeout = dhGlobal::getVal($options,"timeout",true);
        if(is_null(static::$client)) {
            static::init($config);
        }
        $customId = dhGlobal::getVal($options,"customId",null);
        if($destFile !== false) {
            $destFileOrId = $destFile;
            /** @var \boru\dhutils\http\Response */
            $response = static::$client->files()->downloadByName($destFile,true);
        } else {
            $destFileOrId = $fileId;
            /** @var \boru\dhutils\http\Response */
            $response = static::$client->files()->downloadById($fileId,true);
        }
        $destSha1 = $response->header("X-Bz-Content-Sha1")[0];
        if($destSha1 == "none" && is_array($response->header("x-bz-info-large_file_sha1"))) {
            $destSha1 = $response->header("x-bz-info-large_file_sha1")[0];
        }
        //x-bz-info-large_file_sha1
        $sourceSha1 = $file->sha1();
        $uploadDeferred = new Deferred();
        $uploadDeferred->resolve(AsyncWorker::verifyResponse($customId,$sourceFile,$destFileOrId,$sourceSha1,$destSha1));
        return $uploadDeferred->promise();

        
    }
    public static function upload($config=[],$options=[]) {
        if(!is_array($config) || !is_array($options)) {
            throw new \Exception(__CLASS__."::upload - requires 2 args, arg1 is the config array, arg2 is the uploadInfo array",1);
        }
        if(($bucketId = dhGlobal::getVal($options,"bucketId",false)) === false) {
            throw new \Exception(__CLASS__."::upload - missing required option: bucketId",2);
        }
        if(($sourceFile = dhGlobal::getval($options,"sourceFile",false)) === false) {
            throw new \Exception(__CLASS__."::upload - missing required option: sourceFile",2);
        }
        if(($destFile = dhGlobal::getval($options,"destFile",false)) === false) {
            throw new \Exception(__CLASS__."::upload - missing required option: destFile",2);
        }
        if(($file = dhGlobal::fileIfExists($sourceFile)) === false) {
            throw new \Exception(__CLASS__."::upload - sourceFile not a valid file",3);
        }
        $timeout = dhGlobal::getVal($options,"timeout",true);
        if(is_null(static::$client)) {
            static::init($config);
        }
        static::$client->upload()->debugDoTrace=dhGlobal::getval($options,"trace",false);
        $uploadNormalLimit = dhGlobal::getVal($options,"uploadNormalLimit",false);
        $multiPartSize = dhGlobal::getVal($options,"multiPartSize",false);

        //if set, we take the (filesize / uploadNormalLimit) * $partsPerSize as the number of parts, and build
        //the multiPartSize off of that.
        if(($partsPerSize = dhGlobal::getVal($options,"partsPerSize",false)) !== false) {
            $maxParts = floor(($file->size()/$uploadNormalLimit) * $partsPerSize);
            $multiPartSize = min(max(5000000,floor($file->size()/$maxParts)),$uploadNormalLimit);
        }
        $customId = dhGlobal::getVal($options,"customId",null);

        $uploadOptions = [];
        $uploadOptions["bucketId"] = $bucketId;
        $uploadOptions["file"] = $file;
        $uploadOptions["fileName"] = $destFile;
        $uploadOptions["async"] = true;
        $uploadOptions["timeout"] = $timeout;
        if($uploadNormalLimit !== false) {
            $uploadOptions["uploadNormalLimit"] = $uploadNormalLimit;
        }
        if($multiPartSize !== false) {
            $uploadOptions["multiPartSize"] = $multiPartSize;
        }
        $promise = static::$client->upload()->upload($uploadOptions);

        $uploadDeferred = new Deferred();
        $promise->then(function($bucketFileResponse) use (&$uploadDeferred,$sourceFile,$destFile,$customId,&$file) {
            $result = AsyncWorker::uploadResponse($customId,$sourceFile,$destFile,$bucketFileResponse,null,$file);
            $file=null;
            $uploadDeferred->resolve($result);
            
            return $result;
        },function($e) use (&$uploadDeferred,$sourceFile,$destFile,$customId,&$file) {
            $result = AsyncWorker::uploadResponse($customId,$sourceFile,$destFile,null,$e,$file);
            $file=null;
            $uploadDeferred->resolve($result);
        });
        $file=null;
        return $uploadDeferred->promise();
    }

    public static function getClassCallable() {
        return __CLASS__;
    }

    private static function verifyResponse($customId,$sourceFile,$destFileOrId,$sourceSha1,$destSha1) {
        $resultOutput = [
            "customId"=>$customId,
            "success"=>false,
            "source"=>$sourceFile,
            "dest"=>$destFileOrId,
            "sourceSha1"=>$sourceSha1,
            "destSha1"=>$destSha1,
            "errorCode"=>false,
            "error"=>null,
        ];
        try {
            $compare = static::compareSha1($sourceSha1,$destSha1);
            $resultOutput["success"]=true;
        } catch (UploadException $e) {
            $resultOutput["success"]=false;
            $resultOutput["errorCode"] = $e->getType();
            $resultOutput["error"] = $e->getMessage();
        }
        return $resultOutput;
    }

    private static function uploadResponse($customId,$sourceFilePath,$destPath,$bucketFileResponse=false,$errorResponse=false,&$sourceFile=null) {
        $resultOutput = [
            "customId"=>$customId,
            "success"=>false,
            "source"=>$sourceFilePath,
            "dest"=>$destPath,
            "errorCode"=>false,
            "error"=>null,
            "isBucketFile"=>false,
            "uploadResponse"=>null,
        ];

        if(is_object($errorResponse) && method_exists($errorResponse,"getMessage")) {
            $resultOutput["error"] = $errorResponse->getMessage();
        } elseif(!is_object($errorResponse)) {
            $resultOutput["error"] = $errorResponse;
        } else {
            $resultOutput["error"] = "unknown";
        }


        if($bucketFileResponse instanceof BucketFile) {
            $resultOutput["isBucketFile"] = true;
            $resultOutput["uploadResponse"] = $bucketFileResponse->get();
            if(!is_null($bucketFileResponse->fileId()) && !empty($bucketFileResponse->fileId())) {
                $resultOutput["success"]=true;
            } else {
                if($bucketFileResponse->get("sourceSha1",false) !== false) {
                    try {
                        $compare = static::compareSha1($bucketFileResponse->get("sourceSha1",false),$bucketFileResponse->sha1());
                        $resultOutput["success"]=false;
                    } catch (UploadException $e) {
                        $resultOutput["success"]=false;
                        $resultOutput["errorCode"] = $e->getType();
                        $resultOutput["error"] = $e->getMessage();
                    }
                } else {
                    if(is_null($sourceFile)) {
                        $sourceFile = new File(["path"=>$sourceFilePath]);
                    }
                    try {
                        $compare = static::compareSourceToBucketFile($sourceFile,$bucketFileResponse);
                        $resultOutput["success"]=false;
                    } catch (UploadException $e) {
                        $resultOutput["success"]=false;
                        $resultOutput["errorCode"] = $e->getType();
                        $resultOutput["error"] = $e->getMessage();
                    }
                    $sourceFile=null;
                }
            }
        }elseif($bucketFileResponse === false || is_null($bucketFileResponse)) {
            $resultOutput["success"] = false;
            $resultOutput["isBucketFile"] = false;
            $resultOutput["uploadResponse"] = false;
            $resultOutput["errorCode"] = Client::ERR_BLANK_RESPONSE;
        } else {
            $resultOutput["success"] = false;
            $resultOutput["isBucketFile"] = false;
            $resultOutput["uploadResponse"] = $bucketFileResponse;
            $resultOutput["errorCode"] = Client::ERR_NOT_BUCKETFILE;
        }
        $sourceFile=null;
        static::$client= null;
        return $resultOutput;
    }

    private static function compareSourceToBucketFile(File $sourceFile,BucketFile $bucketFile) {
        return static::compareSha1($sourceFile->sha1(),$bucketFile->sha1());
    }
    private static function compareSha1($sourceSha1,$destSha1) {
        if(is_null($sourceSha1) || is_null($destSha1) || empty($sourceSha1) || empty($destSha1)) {
            throw new UploadException(Client::ERR_UPLOAD_SHA1_MISMATCH,"Destination sha1 of $destSha1 != Source sha1 of $sourceSha1");
        }
        if($destSha1 != $sourceSha1) {
            throw new UploadException(Client::ERR_UPLOAD_SHA1_MISMATCH,"Destination sha1 of $destSha1 != Source sha1 of $sourceSha1");
        }
        return true;
    }
}