<?php
namespace boru\boruai\Models\Vectorstore;

use boru\boruai\Config;
use boru\boruai\Models\VectorDocument;
use Predis\Client;
use Predis\Command\Argument\Search\CreateArguments;
use Predis\Command\Argument\Search\SchemaFields\TextField;
use Predis\Command\Argument\Search\SchemaFields\VectorField;
use Predis\Command\Argument\Search\SearchArguments;
use Predis\Response\ServerException;

class RedisVectorStore extends VectorStore {
    private $redis;
    private $redisIndex;

    public function __construct($redis = null, $index = null) {
        $this->redis($redis);
        $this->index($index);
    }

    /**
     * @param Client|null $redis
     * @return Client
     */
    public function redis($redis=null) {
        if ($redis !== null) {
            $this->redis = $redis;
        }
        if($this->redis === null) {
            $this->redis = Config::redis();
        }
        return $this->redis;
    }

    /**
     * @param string|null $index
     * @return string
     */
    public function index($index=null) {
        if ($index !== null) {
            $this->redisIndex = $index;
        }
        if($this->redisIndex === null) {
            $this->redisIndex = Config::vectorPrefix();
        }
        if(substr($this->redisIndex,-1) !== ":") {
            $this->redisIndex.=":";
        }
        return $this->redisIndex;
    }

    public function addDocument(VectorDocument $document) {
        $this->redis->jsonmset(...$this->createJsonSetArgs($document));
    }
    public function addDocuments(array $documents, int $batchSize = 50) {
        if(empty($documents)) {
            return;
        }
        if($batchSize <= 0) {
            $batchSize = count($documents);
        }
        $redisArgs = [];
        $i=0;
        foreach ($documents as $document) {
            array_push($redisArgs, ...$this->createJsonSetArgs($document));
            if($i % $batchSize === 0) {
                $this->redis->jsonmset(...$redisArgs);
                $redisArgs = [];
            }
            $i++;
        }
        if(!empty($redisArgs)) {
            $this->redis->jsonmset(...$redisArgs);
        }
    }

    public function similarity($embedding, $k = 4, $extraArgs = []) {
        $vectorDimension = count($embedding);
        $this->createIndexIfMissing($vectorDimension);
        
        $binaryQueryVector = '';
        foreach ($embedding as $value) {
            $binaryQueryVector .= pack('f', $value);
        }

        $filter = array_key_exists('filters', $extraArgs) ?
            $extraArgs['filters']
            : '*';

        /** @var array{0: int, 1: string, 2: string[]} $redisArray */
        $redisArray = $this->redis->ftsearch(
            $this->index(),
            "($filter)=>[KNN $k @embedding \$query_vector AS distance]",
            (new SearchArguments())
                ->dialect('2')
                ->params(['query_vector', $binaryQueryVector])
                ->sortBy('distance', 'ASC')
        );
        if($redisArray[0] === 0) {
            return [];
        }
        return $this->parseRedisResult($redisArray);
    }

    /**
     * @param  array{0: int, 1: string, 2: string[]}  $redisArray
     * @return VectorDocument[]
     */
    private function parseRedisResult( $redisArray)
    {
        /*
        ### Example of a redisArray ###
            0 => 3                              --> number of results
            1 => "llphant:files:france.txt:0"   --> ($i) redis document id
            2 => array:4 [                      --> ($i + 1)
                0 => "distance"
                1 => "0.121247768402"           --> ($i + 1)[1] distance from the query vector
                2 => "$"
                3 => "{'content':'Fran...       --> ($i + 1)[3] json encoded document
            3 => "llphant:files:paris.txt:0"    --> ($i + 2) = $i on the 2nd iteration of the loop
            4 => array:4 [
                0 => "distance"
                ...
        */

        $documents = [];
        $count = count($redisArray);
        for ($i = 1; $i < $count; $i += 2) {
            [$distanceLabel, $distanceValue, $redisPath, $jsonDocument] = $redisArray[$i + 1];
            $data = json_decode($jsonDocument, true);
            if(is_array($data) && !empty($data)) {
                $documents[] = VectorDocument::fromJson($jsonDocument);
            } else {
                throw new \Exception("Failed to decode json from redis");
            }
        }

        return $documents;
    }

    private function createJsonSetArgs($document) {
        $jsonDocument = json_encode($document->toArray());
        return [
            $this->index().$document->id(),
            '$',
            $jsonDocument
        ];
    }

    private function createIndexIfMissing(int $vectorDimension)
    {
        try {
            $this->redis->ftinfo($this->index());
        } catch (ServerException $e) {
            if ($e->getMessage() !== 'Unknown index name') {
                throw $e;
            }
            $this->createIndex($vectorDimension);
        }
    }
    private function createIndex(int $vectorDimension) {
        $schema = [
            new TextField('$.content', 'content'),
            new TextField('$.formattedContent', 'formattedContent'),
            new TextField('$.hash', 'hash'),
            new TextField('$.name', 'name'),
            new TextField('$.type', 'type'),
            
            new VectorField('$.embedding', 'FLAT', [
                'DIM', $vectorDimension,
                'TYPE', 'FLOAT32',
                'DISTANCE_METRIC', 'COSINE',
            ], 'embedding'),
        ];
        $createArguments = new CreateArguments();
        $createArguments->on('JSON');
        $createArguments->prefix([$this->index()]);
        $this->redis->ftcreate($this->index(), $schema, $createArguments);
    }
}