<?php
namespace boru\boruai\Tools;

use ReflectionClass;
use Exception;

class Util {
    public static function globalIncludeFile($filePath) {
        if (file_exists($filePath)) {
            // Use an anonymous scope to capture the file's variables
            $variables = (function () use ($filePath) {
                include $filePath;
                return get_defined_vars(); // Capture all variables defined in the included file
            })();
    
            // Extract the variables into the global scope
            foreach ($variables as $key => $value) {
                $GLOBALS[$key] = $value;
            }
        } else {
            throw new Exception("File '$filePath' does not exist.");
        }
    }
    public static function getClassFileContents(string $className) {
        try {
            echo "Getting class file contents for '$className'..." . PHP_EOL;
            // Resolve class name if it doesn't exist in the global namespace
            if (!class_exists($className) && !interface_exists($className) && !trait_exists($className)) {
                //if $className contains a single backslash, then we need to double it before resolving
                if(strpos($className, "\\") !== false) {
                    $className = str_replace("\\", "\\\\", $className);
                }
                $className = static::resolveNamespacedClassName($className);
            }

            echo "Resolved class name: '$className'" . PHP_EOL;
            // Check again if the resolved class exists
            if (!class_exists($className) && !interface_exists($className) && !trait_exists($className)) {
                throw new Exception("Class, interface, or trait '$className' does not exist.");
            }

            // Collect file contents from the given class and its dependencies
            $collectedFiles = [];

            // Recursive function to gather class/trait/interface files
            $collectFiles = function ($name) use (&$collectedFiles, &$collectFiles) {
                $reflection = new ReflectionClass($name);
                $filePath = $reflection->getFileName();

                if ($filePath && file_exists($filePath) && !in_array($filePath, $collectedFiles)) {
                    $collectedFiles[] = $filePath;
                }

                // Collect parent class
                $parentClass = $reflection->getParentClass();
                if ($parentClass) {
                    $collectFiles($parentClass->getName());
                }

                // Collect traits
                $traits = $reflection->getTraits();
                foreach ($traits as $trait) {
                    $collectFiles($trait->getName());
                }

                // Collect interfaces
                $interfaces = $reflection->getInterfaces();
                foreach ($interfaces as $interface) {
                    $collectFiles($interface->getName());
                }
            };

            // Start collecting files from the main class
            $collectFiles($className);

            // Concatenate the contents of all collected files
            $allContents = "";
            foreach ($collectedFiles as $file) {
                $allContents .= "\n--- File: $file ---\n";
                $allContents .= file_get_contents($file);
            }

            return $allContents;
        } catch (Exception $e) {
            // Handle any exceptions and return null if something goes wrong
            echo "Error: " . $e->getMessage() . PHP_EOL;
            return null;
        }
    }

    /**
     * Resolves a class name to its fully qualified name if it is namespaced.
     * @param string $className 
     * @return string 
     */
    public static function resolveNamespacedClassName(string $className) {
        foreach (get_declared_classes() as $declaredClass) {
            if (substr($declaredClass, -strlen($className)) === $className && 
                (strpos($declaredClass, "\\") !== false)) {
                return $declaredClass;
            }
        }

        foreach (get_declared_interfaces() as $declaredInterface) {
            if (substr($declaredInterface, -strlen($className)) === $className && 
                (strpos($declaredInterface, "\\") !== false)) {
                return $declaredInterface;
            }
        }

        foreach (get_declared_traits() as $declaredTrait) {
            if (substr($declaredTrait, -strlen($className)) === $className && 
                (strpos($declaredTrait, "\\") !== false)) {
                return $declaredTrait;
            }
        }

        return $className; // Fallback to the original name
    }
}