<?php
namespace boru\boruai\Vector;

use boru\boruai\BoruAI;
use boru\boruai\Models\Embedding;
use boru\output\Output;
use boru\qdrant\Models\Collection;
use boru\qdrant\Models\Point;
use boru\query\Entity;
use boru\query\models\EntityDefinition;
use boru\query\Query;

class VectorCollection {
    /** @var Collection */
    private $collection;

    private $recordClassName;

    public function __construct($initOptions=[]) {
        $this->fromOptions($initOptions);
        if($this->collection) {
            $this->collection->save();
        }
    }
    public function collection($collection=null,$vector=null,$options=[]) {
        if ($collection !== null) {
            if($collection instanceof Collection) {
                $this->collection = $collection;
            } else {
                $this->collection = new Collection($collection,$vector,$options);
            }
            $this->collection = $collection;
            return $this;
        }
        return $this->collection;
    }
    public function name() {
        if($this->collection) {
            return $this->collection->name();
        }
        return null;
    }
    public function recordClassName($recordClassOrObject=null) {
        if($recordClassOrObject instanceof CollectionRecord) {
            $this->recordClassName = get_class($recordClassOrObject);
        } elseif(is_string($recordClassOrObject)) {
            $this->recordClassName = $recordClassOrObject;
        }
        if($this->recordClassName === null) {
            $this->recordClassName = CollectionRecord::class;
        }
        return $this->recordClassName;
    }

    public function info() {
        if($this->collection) {
            //return $this->collection->info();
        }
        return null;
    }

    /**
     * 
     * @param CollectionRecord $collectionRecord 
     * @return bool
     * @throws \Exception
     */
    public function upsert($collectionRecord) {
        $point = null;
        if($collectionRecord instanceof CollectionRecord) {
            $point = $collectionRecord->point();
        } elseif($collectionRecord instanceof Point) {
            $point = $collectionRecord;
        } 
        if($point === null) {
            throw new \Exception("Invalid point type");
        }
        if($this->collection) {
            return $this->collection->addPoint($point);
        }
        throw new \Exception("Collection not set");
    }

    private function fromOptions($options=[]) {
        $vectorParams = $collectionName = null;
        if(!is_array($options) && is_string($options)) {
            $check = json_decode($options,true);
            if($check) {
                $options = $check;
            }
        }
        if(is_array($options)) {
            if(isset($options["vectorParams"])) {
                $vectorParams = $options["vectorParams"];
                unset($options["vectorParams"]);
            } elseif(isset($options["vector"])) {
                $vectorParams = $options["vector"];
                unset($options["vector"]);
            }
            if(isset($options["collectionName"])) {
                $collectionName = $options["collectionName"];
                unset($options["collectionName"]);
            } elseif(isset($options["collection"])) {
                $collectionName = $options["collection"];
                unset($options["collection"]);
            }
        } elseif($options instanceof Collection) {
            $collectionName = $options;
        } elseif(is_string($options)) {
            $collectionName = $options;
        }
        if($collectionName && $collectionName instanceof Collection) {
            $this->collection = $collectionName;
        } elseif($collectionName) {
            $this->collection = new Collection($collectionName,$vectorParams,$options);
        }
    }

    /**
     * @param string $id
     * @return CollectionRecord|null
     */
    public function retrieve($id) {
        if($this->collection) {
            try {
                $point = $this->collection->getPoint($id);
                if($point) {
                    return CollectionRecord::fromPoint($point);
                }
            } catch (\Exception $e) {
                Output::outLine("Error retrieving point: " . $e->getMessage());
            }
        }
        return null;
    }

    /**
     * 
     * @param string|CollectionRecord|Point $idOrRecord 
     * @return boru\qdrant\ApiBaseResponse|false 
     */
    public function delete($idOrRecord) {
        if($this->collection) {
            if($idOrRecord instanceof CollectionRecord) {
                $id = $idOrRecord->uid();
            } elseif($idOrRecord instanceof Point) {
                $id = $idOrRecord->id();
            } elseif(is_string($idOrRecord)) {
                $id = $idOrRecord;
            } else {
                return false;
            }
            try {
                return $this->collection->deletePoint($id);
            } catch (\Exception $e) {
                Output::outLine("Error deleting point: " . $e->getMessage());
            }
        }
        return false;
    }


    public function search($pointRecordOrString,$filter=[],$limit=10,$withPayload=true,$withVector=false) {
        if($this->collection()) {
            $vector = null;
            $className = $this->recordClassName();
            if($pointRecordOrString instanceof CollectionRecord) {
                $point = $pointRecordOrString->point();
                $vector = $pointRecordOrString->vector();
                $className = get_class($pointRecordOrString);
            } elseif($pointRecordOrString instanceof Point) {
                $point = $pointRecordOrString;
                $vector = $point->vector();
            } elseif(is_array($pointRecordOrString)) {
                $vector = $pointRecordOrString;
            } elseif(is_string($pointRecordOrString)) {
                $vector = Embedding::embed($pointRecordOrString);
            }
            if($vector) {
                $points = $this->collection->search($vector,$filter,$limit,$withPayload,$withVector);
                if($points) {
                    $records = [];
                    foreach($points as $point) {
                        $record = $className::fromPoint($point);
                        if($record) {
                            $records[] = $record;
                        }
                    }
                    return $records;
                }
            }
        }
        return [];
    }
}