<?php
namespace boru\boruai\Models;

use boru\boruai\BoruAI;
use boru\boruai\Openai\Api\Endpoints\ResponsesAPI;
use boru\boruai\Openai\Response\Parts\FunctionCall;
use boru\boruai\Openai\Response\ResponseObject;
use boru\dhutils\tools\Output;
use boru\boruai\Openai\Stream\StreamEvent;
use boru\query\models\Value;
use boru\query\Query;
use GuzzleHttp\Psr7\Response as GuzzleResponse;

class Response extends ResponseObject {
    /** @var int */
    private $iterations = 0;
    /** @var StreamEvent[] */
    private $events = [];

    /** @var string */
    private $partialLine = "";

    /** @var callable */
    private $callback = null;

    /** @var ResponseObject */
    private $response;

    private $saveHistory = true;
    private $historyRecord = null;

    private $eventCallbacks = [];

    private $reference = null;
    private $referenceUser;
    private $templateId;
    private $responseGroup;

    private $throwExceptionOnError = true;

    public function __construct($input=null) {
        if(!is_null($input)) {
            if(is_array($input)) {
                if(isset($input['reference'])) {
                    $this->reference($input['reference']);
                    unset($input['reference']);
                }
            } else {
                if(!is_object($input)) {
                    $input = json_decode($input,true);
                    if(!is_array($input)) {
                        $input = [];
                    }
                }
            }
            parent::__construct($input);
        }
    }

    public function callback($callback=null) {
        if ($callback === null) {
            $this->callback = null;
        }
        if (is_callable($callback)) {
            $this->callback = $callback;
        } else {
            throw new \Exception("Callback is not callable");
        }
        return $this;
    }

    public function throwExceptionOnError($throw=null) {
        if($throw !== null) {
            $this->throwExceptionOnError = $throw ? true : false;
        }
        return $this->throwExceptionOnError ? true : false;
    }

    public function saveHistory($save=null) {
        if($save !== null) {
            $this->saveHistory = $save;
        }
        return $this->saveHistory ? true : false;
    }
    public function getHistoryRecord($responseId=null) {
        if($this->historyRecord) {
            return $this->historyRecord;
        }
        if($responseId) {
            $this->historyRecord = ResponseHistory::fromResponseId($responseId);
            if(!$this->historyRecord) {
                $this->historyRecord = new ResponseHistory();
                $this->historyRecord->responseId($responseId);
            }
        }
        if($this->historyRecord) {
            $this->historyRecord->model($this->model());
            $this->historyRecord->input($this);
        }
        return $this->historyRecord;

    }

    public function on($eventType, $callback) {
        if (is_callable($callback)) {
            $eventType = strtolower($eventType);
            $this->eventCallbacks[$eventType][] = $callback;
        } else {
            throw new \Exception("Callback is not callable");
        }
        return $this;
    }

    public function events() {
        return $this->events;
    }
    public function reference($reference=null) {
        if($reference !== null) {
            $this->reference = $reference;
        }
        return $this->reference;
    }
    public function referenceUser($user=null) {
        if($user !== null) {
            $this->referenceUser = $user;
        }
        return $this->referenceUser;
    }
    public function templateId($id=null) {
        if($id !== null) {
            $this->templateId = $id;
        }
        return $this->templateId;
    }
    public function response() {
        return $this->response;
    }
    public function getResult($asString=true) {
        if($this->response) {
            return $this->response->getResult($asString);
        }
        return null;
    }

    public function setResponse($response) {
        if($response instanceof ResponseObject) {
            $this->response = $response;
        } else {
            throw new \Exception("Response is not a ResponseObject");
        }
        if($this->saveHistory()) {
            $this->getHistoryRecord($response->id());
            if($this->historyRecord) {
                $this->historyRecord->output($response);
                if($response->error()) {
                    $error = $response->error();
                    if(!is_array($error)) {
                        $tryError = json_decode($error,true);
                        if($tryError) {
                            $error = $tryError;
                        }
                    }
                    if(is_array($error)) {
                        $this->historyRecord->result($error["message"]);
                    } else {
                        $this->historyRecord->result($error);
                    }
                }
                //$this->historyRecord->result($guzzleArray["error"]["message"]);
                
                $this->historyRecord->reference($this->reference());
                $this->historyRecord->user($this->referenceUser());
                $this->historyRecord->templateId($this->templateId());
                $this->historyRecord->save();
            }
        }
    }

    /**
     * 
     * @param StreamEvent $event 
     * @return void
     */
    public function processEventCallback($event) {
        $this->parseResponseObjectEvent($event);
        if($this->callback && is_callable($this->callback)) {
            $callback = $this->callback;
            $callback($event);
        }
        /*if($event->type() == "response.completed" && $event instanceof \boru\boruai\Openai\Stream\Response\Completed) {
            $obj = $event->response();
            if(!$obj instanceof ResponseObject) {
                $obj = new ResponseObject($obj);
            }
            $this->setResponse($obj);
        }*/
        $etLower = strtolower($event->type());
        if(isset($this->eventCallbacks[$etLower])) {
            foreach($this->eventCallbacks[$etLower] as $callback) {
                if(is_callable($callback)) {
                    $callback($event);
                }
            }
        }
    }
    private function parseResponseObjectEvent($event) {
        if(is_object($event) && method_exists($event, "response")) {
            $obj = $event->response();
            if(!$obj instanceof ResponseObject) {
                $obj = new ResponseObject($obj);
            }
            $this->setResponse($obj);
        }
    }

    /**
     * @param guzzleResponse
     */
    public function processStream($guzzleResponse) {
        $this->events = [];
        $this->partialLine = "";
        $previousResponse = $this->response;
        $this->response = null;
        //$this->handleStream($guzzleResponse);
        if($guzzleResponse->getStatusCode() !== 200) {
            $headers = $guzzleResponse->getHeaders();
            //print error?
            $guzzleContent = $guzzleResponse->getBody()->getContents();
            $guzzleArray = json_decode($guzzleContent, true);
            if($guzzleArray && isset($guzzleArray['error'])) {
                //all set
                $guzzleArray['error']['content'] = $guzzleContent;
            } else {
                $guzzleArray = [
                    'error' => [
                        'message' => "Unknown error",
                        'type' => "unknown",
                        'code' => $guzzleResponse->getStatusCode(),
                        "content" => $guzzleContent,
                    ]
                ];
                if(is_array($guzzleArray)) {
                    $guzzleArray['error']['content'] = $guzzleArray;
                }
            }
            $guzzleArray['error']['headers'] = $headers;
            if(!isset($guzzleArray['id'])) {
                $guzzleArray['id'] = "error-".time()."_".uniqid();
            }
            $this->makeEvent($guzzleArray, "response.error");
            $this->setResponse(new ResponseObject($guzzleArray));
            if($this->throwExceptionOnError) {
                throw new \Exception("Error in response: ".$guzzleArray['error']['message'], $guzzleArray['error']['code']);
            }
            return false;
        }
        $body = $guzzleResponse->getBody();
        while(!$body->eof()) {
            $lines = trim($body->read(1024));
            $this->parseLines($lines);
        }
        $this->processLine();
    }
    public function parseLines($lineString) {
        $lines = explode("\n",$lineString);
        foreach($lines as $line) {
            if(trim($line) == "") {
                continue;
            }
            if (strpos($line, 'data: ') === 0) {
                $this->processLine();
                $this->partialLine = "";
                $line = substr($line, 6);
            }
            if (strpos($line, 'event: ') === 0) {
            } else {
                $this->partialLine .= $line;
            }
        }
    }

    public function processLine() {
        $line = $this->partialLine;
        if(empty($line)) {
            return;
        }
        $data = json_decode($line, true);
        if($data && isset($data['type'])) {
            $this->makeEvent($data);
        }
    }

    public function initResponseGroup() {
        if($this->responseGroup) {
            return $this->responseGroup;
        }
        if($this->reference() || $this->referenceUser()) {
            $this->responseGroup = ResponseGroup::forReference($this->reference(),$this->referenceUser());
            if(!$this->instructions() && $this->responseGroup->instructions()) {
                $this->instructions($this->responseGroup->instructions());
            }
            //$this->addToInput($this->responseGroup->getInput());
        }
    }



    public function create($input=null,$callback=null,$stream=true) {
        $this->iterations = 0;
        if($callback && is_callable($callback)) {
            $this->callback($callback);
        }
        if ($input) {
            $this->addMessage($input);
        }
        if(!$this->model()) {
            $this->model(BoruAI::defaultModel("default","gpt-4.1"));
        }
        $this->initResponseGroup();
        $result = $this->executeStream($stream);
        if($result && $this->saveHistory()) {
            $this->getHistoryRecord($result->response()->id());
            if($this->historyRecord) {
                $this->historyRecord->output($result->response());
                $this->historyRecord->result($result->getResult());
                $this->historyRecord->reference($this->reference());
                $this->historyRecord->user($this->referenceUser());
                $this->historyRecord->templateId($this->templateId());
                $this->historyRecord->input($this);
                $this->historyRecord->save();
            }            
        }
        return $result;
    }
    /**
     * 
     * @return self|false
     * @throws Exception 
     */
    private function executeStream($stream=true) {
        if(!$stream) {
            $response = ResponsesAPI::create($this->forInput(),false);
            if($response instanceof ResponseObject) {
                $this->setResponse($response);
                $result = $this->response();
            }
        } else {
            try {
                $streamedResult = ResponsesAPI::create($this->forInput(),true);
                $this->processStream($streamedResult);
                $result = $this->response();
            } catch (\Exception $e) {
                Output::outLine("Requested with:",$this->forInput());
                Output::outLine("Error: ",$e->getMessage());
                throw $e;
                $result = false;
            }
        }
        if($result) {
            $output = $result->output();
            if(!is_array($output)) {
                $output = [$output];
            }
            $haveFunctionCalls = false;
            foreach($output as $message) {
                if($message instanceof FunctionCall) {
                    $this->addToInput($message);
                    $functionData = $message->toArray();
                    $functionData['type'] = 'function_call.started';
                    $this->makeEvent($functionData);
                    //Output::outLine("Function Call ID: ",$message->id());
                    //Output::outLine("Function Call Name: ",$message->name());
                    $data = $message->run();
                    if($data) {
                        //Output::outLine("Function Call Data: ",$data);
                        $haveFunctionCalls = true;
                        $this->addToInput($data);
                        $functionData['type'] = 'function_call.completed';
                        $functionData['output'] = isset($data["output"]) ? $data["output"] : null;
                        $this->makeEvent($functionData);
                    } else {
                        $functionData['type'] = 'function_call.failed';
                        $this->makeEvent($functionData);
                    }
                }
            }
            if($haveFunctionCalls) {
                $this->toolChoice("auto");
                return $this->executeStream($stream);
            }
            return $this;
        }
        return false;
    }
    private function makeEvent($data = [], $type = null) {
        $eventData = $data;
        if (is_string($data)) {
            $eventData = json_decode($data, true);
        }
        if($type !== null) {
            $eventData['type'] = $type;
        }
        $event = StreamEvent::fromEvent($eventData,[$this, "processEventCallback"]);
        if($event) {
            $this->events[] = $event;
            return;
        }
    }
    public static function fromHistoryId($id) {
        $history = ResponseHistory::fromId($id);
        if($history) {
            $response = $history->get("input");
            if($response instanceof ResponseObject) {
                return $response;
            } else {
                $response = new Response($response);
                return $response;
            }
        }
        return false;
    }
    public static function fromId($id) {
        return ResponsesAPI::get($id);
    }
    public static function fromResponseHistory($history) {
        if($history instanceof ResponseHistory) {
            $response = $history->input();
            if($response instanceof ResponseObject) {
                return $response;
            } else {
                $response = new Response($response);
                return $response;
            }
        } elseif(is_numeric($history)) {
            $history = ResponseHistory::fromId($history);
            if($history) {
                return self::fromResponseHistory($history);
            }
        } elseif(is_string($history)) {
            $history = ResponseHistory::fromId($history);
            if($history) {
                return self::fromResponseHistory($history);
            }
        }
        return false;
    }
    public static function fromReference($reference,$user=null) {
        $instance = self::withReference($reference,$user);
        if($instance) {
            return $instance;
        }
        return false;
    }
    public static function withReference($reference,$user=null,$createIfNotExists=true) {
        $db = BoruAI::db();
        //$db->printDebug(true);
        $tableName = BoruAI::table('responses');
        $query = Query::create()->select("id","model","response_id","template_id")->from($tableName);
        $query->where("reference","=",$reference)->where(Value::functionValue("LEFT(response_id,3)"),"!=","err")->where("result","!=","");
        if($user) {
            if(is_numeric($user)) {
                $query->where("user","=",$user);
            }
        }
        $query->limit(0,1)->orderBy("id","desc");
        $result = $query->toRow();
        if($result) {
            $responseId = $result->get("response_id");
            $model = $result->get("model");
            $templateId = $result->get("template_id");
            $instance = new self();
            $instance->previousResponseId($responseId);
            $instance->model($model);
            $instance->reference($reference);
            if($templateId) {
                $instance->templateId($templateId);
            }

            if($user) {
                if(is_numeric($user)) {
                    $instance->referenceUser($user);
                }
            }
            return $instance;
        } elseif($createIfNotExists) {
            $instance = new self();
            $instance->reference($reference);
            if($user) {
                if(is_numeric($user)) {
                    $instance->referenceUser($user);
                }
            }
            return $instance;
        }
        return false;
    }
    public static function fromResponse($response,$copyId=false) {
        if(!$response instanceof ResponseObject) {
            if(is_array($response)) {
                $response = new Response($response);
            } else {
                $response = Response::fromId($response);
            }
        }
        if(!$response instanceof ResponseObject) {
            throw new \Exception("Invalid response object");
        }
        $array = $response->toArray();
        if(!$copyId) {
            unset($array["id"]);
        }
        $instance = new self($array);
        if(!$copyId) {
            $instance->previousResponseId($response->id());
        }
        return $instance;
    }
    public static function withTemplate($template,$data=[]) {
        if($template instanceof Template) {
            $template = $template->id();
        }
        if(is_numeric($template)) {
            $instance = new self();
            $instance->templateId($template);
            if(is_array($data)) {
                foreach($data as $key => $value) {
                    $instance->addMessage($key,$value);
                }
            }
            return $instance;
        }
        return false;
    }
};