<?php
namespace boru\boruai\Socket;

/**
 * Handle a socket event.
 * This class is used to encapsulate the event name and data, and a callback function that will be executed when the event is triggered.
 * It is designed to be extended by specific event handlers that implement the logic for handling the event
 * @package boru\boruai\Socket
 */
abstract class SocketHandler {

    private $socketListener;

    public static function register(SocketListener $socketListener) {
        $eventName = static::eventName();
        $socketListener->on($eventName, [static::class, 'handleEvent']);
    }

    public static function handleEvent($socketListener, $data) {
        $handler = new static($socketListener);
        return $handler->handle($data);
    }

    /**
     * Get the name of the event that this handler listens to.
     * @return string The name of the event.
     * This should be overridden by the subclass to return the specific event name.
     * @example
     * return 'user.login';
     */
    abstract public static function eventName();
    
    /**
     * Constructor for the SocketHandler.
     * @param SocketListener $socketListener The socket listener instance that this handler will use to listen for events.
     */
    public function __construct(SocketListener $socketListener) {
        $this->socketListener = $socketListener;
    }

    public function sendSocket($event, $target, $payload) {
        $socketAPI = SocketAPI::getInstance();
        return $socketAPI->api_event($event, $target, $payload);
    }

    /**
     * Handle the event with the provided data.
     * This method should be implemented by subclasses to define the specific logic for handling the event.
     * @param mixed $data The data associated with the event.
     * @return void
     * This method does not return any value, it is expected to perform actions based on the event data.
     */
    abstract public function handle($data);
}