<?php
namespace boru\boruai\Models\Tool;

use boru\boruai\BoruAI;
use boru\boruai\Models\ToolDefinition;

class McpTool extends ToolDefinition {
    private $type = "mcp";
    private $server_url = "";
    private $server_label = "php-mcp";
    private $allowed_tools = [];
    private $require_approval = "never";
    private $headers = [];

    public function __construct($input = [], $rawJson = false) {
        $name = isset($input["name"]) ? $input["name"] : "mcp_".date("YmdHis");
        $this->name($name);
        $this->description("A tool to interact with the MCP system (new format).");
        $this->type = isset($input["type"]) ? $input["type"] : "mcp";
        // New-format fields
        $this->server_url = isset($input["server_url"]) ? $input["server_url"] : BoruAI::mcpUrl();
        $this->server_label = isset($input["server_label"]) ? $input["server_label"] : $this->server_label;
        $this->allowed_tools = isset($input["allowed_tools"]) ? (array)$input["allowed_tools"] : [];
        $this->require_approval = isset($input["require_approval"]) ? $input["require_approval"] : $this->require_approval;
        $this->headers = isset($input["headers"]) ? (array)$input["headers"] : [];

        // Backwards-compatibility: translate some old-format fields if provided
        if (isset($input["port"]) && !isset($this->headers["internal-port"])) {
            $this->headers["internal-port"] = (string)$input["port"];
        }
        if (isset($input["host"]) && empty($this->server_url)) {
            // best-effort conversion to a server_url if host/port provided
            $port = isset($input["port"]) ? $input["port"] : (isset($this->headers["internal-port"]) ? $this->headers["internal-port"] : "9000");
            $this->server_url = "http://{$input['host']}:{$port}/mcp";
        }
        if (isset($input["tool_auth"]) && !isset($this->headers["Authorization"])) {
            // tool_auth could be array with strategy/token or a string token
            if (is_array($input["tool_auth"])) {
                $strategy = isset($input["tool_auth"]["strategy"]) ? strtolower($input["tool_auth"]["strategy"]) : "bearer";
                $token = isset($input["tool_auth"]["token"]) ? $input["tool_auth"]["token"] : null;
                if ($token !== null) {
                    $prefix = ($strategy === "bearer") ? "Bearer " : "";
                    $this->headers["Authorization"] = $prefix . $token;
                }
            } elseif (is_string($input["tool_auth"])) {
                $this->headers["Authorization"] = $input["tool_auth"];
            }
        }
    }

    public function forResponse() {
        $data = [
            "type" => $this->type,
            "server_url" => $this->server_url,
            "server_label" => $this->server_label,
            "allowed_tools" => $this->allowed_tools,
            "require_approval" => $this->require_approval,
            "headers" => $this->headers,
        ];
        $data = array_filter($data);
        return $data;
    }

    public function compile($asJson = false) {
        $data = [
            "type" => $this->type,
            "server_url" => $this->server_url,
            "server_label" => $this->server_label,
            "allowed_tools" => $this->allowed_tools,
            "require_approval" => $this->require_approval,
            "headers" => $this->headers,
        ];
        $data = array_filter($data);
        if ($asJson) {
            return json_encode($data, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
        }
        return $data;
    }

    /** Compatibility, but does nothing */
    public function regisster($callback = null) {
        return;
    }

    public static function fromJson($jsonString) {
        $data = is_array($jsonString) ? $jsonString : json_decode($jsonString, true);
        return new McpTool($data);
    }
}
