// ChatService: handles SSE, state, and server communication
class ChatService {
    constructor(basePath) {
        this.basePath = basePath;
        this.state = { queue: [], processing: false, openItems: new Set() };
        this.handlers = {};
    }

    on(type, fn) { this.handlers[type] = fn; }
    emit(ev) { if (this.handlers[ev.type]) this.handlers[ev.type](ev); }

    async postAndListen(data) {
        const url = `${this.basePath}index.php?action=chat`;
        const opts = { method: 'POST', headers: { 'Content-Type': 'application/json' }, body: JSON.stringify(data) };
        const res = await fetch(url, opts);
        const reader = res.body.getReader();
        const dec = new TextDecoder();
        let buf = '';
        while (true) {
            const { value, done } = await reader.read();
            if (done) break;
            buf += dec.decode(value, { stream: true });
            const parts = buf.split(/\r?\n\r?\n/);
            buf = parts.pop();
            parts.forEach(p => p.split(/\r?\n/).forEach(line => {
                if (line.startsWith('data:')) {
                    try {
                        const ev = JSON.parse(line.replace(/^data:\s*/, ''));
                        this.emit(ev);
                    } catch { }
                }
            }));
        }
        reader.releaseLock();
    }

    /*schedule() {
        if (this.state.processing) return;
        this.state.processing = true;
        requestAnimationFrame(() => this.flush());
    }
    flush() {
        this.state.processing = false;
        this.state.queue.forEach(ev => this.emit(ev));
        this.state.queue = [];
    }*/

    sendMessage(msg) { return this.postAndListen({ message: msg, reference: $("#sessionReference").val(), instructions: $("#chatInstructions").html() }); }
    /**
     * @param {string} ref       session reference
     * @param {string|null} before   ID (or timestamp) of the oldest message you have – fetch older than this
     * @param {number} limit     how many messages to fetch
     */
    async fetchHistory(ref, before = null, limit = 20) {
        let url = `${this.basePath}index.php?action=history&reference=${encodeURIComponent(ref)}&limit=${limit}`;
        if (before) url += `&before=${encodeURIComponent(before)}`;
        const res = await fetch(url);
        return res.json();  // expects { success, data: { output: [ … ] } }
    }
    async fetchResponse(id) {
        const res = await fetch(`${this.basePath}index.php?action=response&responseId=${id}`);
        return res.json();
    }
}