<?php
namespace boru\boruai\OCR;

use boru\boruai\BoruAI;
use boru\queue\Queue;
use boru\queue\Entity\QueueItem;
use boru\queue\Task\ClosureTask;
use boru\queue\Task\TaskRegistry;

class VtigerDocument {


    /**
     * @var string Default field to update in vtiger with OCR data
     */
    public static $defaultUpdateField = 'cf_ocr_data';
    
    /**
     * @var int Document CRM Id
     */
    public $id;
    public $filepath;
    public $md5hash;
    public $filesize;

    public $updateField;

    public function __construct($data = [], $updateField = null) {
        if($updateField === null) {
            $updateField = self::$defaultUpdateField;
        }
        $this->updateField = $updateField;
        foreach ($data as $key => $value) {
            if (property_exists($this, $key)) {
                $this->$key = $value;
            }
        }
        if(!$this->md5hash && $this->filepath && is_file($this->filepath)) {
            $this->md5hash = md5_file($this->filepath);
        }
    }

    public function toArray() {
        return [
            'id' => $this->id,
            'filepath' => $this->filepath,
            'md5hash' => $this->md5hash,
            'filesize' => $this->filesize,
            'updateField' => $this->updateField,
        ];
    }

    public function payload() {
        return $this->toArray();
    }

    /**
     * Load a VtigerDocument object for a given payload
     * @param mixed $payload 
     * @return VtigerDocument 
     */
    public static function fromPayload($payload) {
        if(!is_array($payload) && !is_object($payload)) {
            if(is_string($payload)) {
                $payload = json_decode($payload, true);
            } else {
                if(is_object($payload) && method_exists($payload, 'toArray')) {
                    $payload = $payload->toArray();
                } else {
                    $payload = [];
                }
            }
        }
        if(!is_array($payload)) {
            throw new \Exception("Invalid payload for VtigerDocument");
        }
        $payloadField = static::$defaultUpdateField;
        if(isset($payload["updateField"])) {
            $payloadField = $payload["updateField"];
            unset($payload["updateField"]);
        }
        return new self($payload, $payloadField);
    }

    /**
     * Load a VtigerDocument by its document CRM Id
     * @param int $docid 
     * @param string|null $updateField
     * @return VtigerDocument|null 
     */
    public static function fromDocumentId($docid, $updateField = null, $includeIfNoFile=false) {
        $sql = "SELECT  vtiger_attachments.* , vtiger_notes.title,vtiger_notes.filename,vtiger_crmentity.createdtime , vtiger_notes.*
        FROM vtiger_notes
        INNER JOIN vtiger_senotesrel ON vtiger_senotesrel.notesid= vtiger_notes.notesid
        INNER JOIN vtiger_crmentity ON vtiger_crmentity.crmid= vtiger_notes.notesid AND vtiger_crmentity.deleted=0
        LEFT JOIN vtiger_seattachmentsrel  ON vtiger_seattachmentsrel.crmid =vtiger_notes.notesid
        LEFT JOIN vtiger_attachments ON vtiger_seattachmentsrel.attachmentsid = vtiger_attachments.attachmentsid
        WHERE vtiger_notes.notesid = ?";
        $db = \boru\boruai\BoruAI::db();
        $result = $db->run($sql, [$docid]);
        while($row=$db->next($result)) {
            $filepath = $row['path'].$row['attachmentsid']."_".$row['name'];
            if(is_file($filepath) || $includeIfNoFile) {
                $data = [
                    'id' => $row['notesid'],
                    'filepath' => $filepath,
                    'filesize' => is_file($filepath) ? filesize($filepath) : null,
                ];
                return new self($data, $updateField);
            }
        }
    }

    /**
     * @param Queue $queue 
     * @return void 
     */
    public function queue($queue) {
        $queueItem = $queue->enqueue("vtiger_document_ocr", $this->toArray(), $this->id);
        if(!$this->filepath || !is_file($this->filepath)) {
            $storage = $queue->getStorage();
            $storage->markError($queueItem, "File does not exist: ".$this->filepath);
            return false;
        }
        return true;
    }

    public function ocr() {
        if(!$this->filepath || !is_file($this->filepath)) {
            throw new \Exception("File does not exist: ".$this->filepath);
        }
        $ocr = new OCR($this->filepath, "DocumentID:".$this->id);
        $result = $ocr->run("DocumentID:".$this->id);
        $db = BoruAI::db();
        $ocrData = $result;
        if(is_array($result)) {
            $ocrData = json_encode($result, JSON_PRETTY_PRINT|JSON_UNESCAPED_SLASHES|JSON_UNESCAPED_UNICODE);
        }
        if($this->updateField !== false && $this->id) {
            $sql = "UPDATE vtiger_notes
                LEFT JOIN vtiger_notescf ON vtiger_notescf.notesid= vtiger_notes.notesid
                INNER JOIN vtiger_crmentity ON vtiger_crmentity.crmid= vtiger_notes.notesid
                SET ".$this->updateField." = ? WHERE notesid = ?";
            $db->run($sql, [$ocrData, $this->id]);
        }
        return $ocrData;
    }

    /**
     * 
     * @param Queue $queue 
     * @param string $taskName 
     * @return void 
     */
    public static function registerTask($queue, $taskName = 'vtiger_document_ocr') {
        $queue->registerTask($taskName, function($queueItem, $payload) {
            $payload = $queueItem->getPayload();
            return self::processOcr($payload);
        });
    }

    public static function processOcr($payload) {
        $document = self::fromPayload($payload);
        return $document->ocr();
    }
}