// ChatInputUI: handles user input, drag/drop, paste, and send events
class ChatInputUI {
    constructor(service, ui) {
        this.service = service;
        this.ui = ui;
        window.uploadedFiles = window.uploadedFiles || [];
        this.$msg = $('#userMessage');
        this.$send = $('#sendMessage');
        this.$preview = $('#previewArea');
    }

    init() {
        this.bindDragDrop();
        this.$send.on('click', () => this.handleSend());
        //this.$msg.on('keypress', e => e.which === 13 && this.handleSend());
        this.$msg.on('keydown', e => {
            if (e.key === 'Enter' && !e.shiftKey) {
              e.preventDefault();
              this.handleSend();
            }
        });
    }

    bindDragDrop() {
        this.$msg
            .on('paste', e => {
                const items = (e.originalEvent.clipboardData || {}).items || [];
                // only intercept if there's a file
                const hasFile = Array.from(items).some(it => it.kind === 'file');
                if (!hasFile) return;            // let text paste through
                e.preventDefault();
                this.handleItems(items);
            })
            .on('drop', e => {
                e.preventDefault();
                const items = (e.originalEvent.dataTransfer || {}).items || [];
                this.handleItems(items);
            })
            .on('dragover', e => e.preventDefault());

        this.$preview.on('click', '.remove-file', e => {
            const id = $(e.currentTarget).data('id'); this.removeFile(id);
        });
    }

    bindDragDrop_old() {
        this.$msg.on('paste drop', e => {
            e.preventDefault();
            const items = (e.originalEvent.clipboardData || e.originalEvent.dataTransfer).items;
            this.handleItems(items);
        }).on('dragover', e => e.preventDefault());

        this.$preview.on('click', '.remove-file', e => {
            const id = $(e.currentTarget).data('id'); this.removeFile(id);
        });
    }

    handleItems(items) {
        Array.from(items).forEach(item => {
            if (item.kind === 'file') {
                const file = item.getAsFile();
                const reader = new FileReader();
                reader.onload = e => this.addPreview(file, e.target.result);
                reader.readAsDataURL(file);
            }
        });
    }

    addPreview(file, data) {
        const id = 'file-' + Date.now() + Math.random().toString(36).substr(2, 6);
        window.uploadedFiles.push({ id, file });
        const isImg = file.type.startsWith('image/');
        const tpl = isImg
            ? `<div id="${id}" class="file-preview"><img src="${data}" style="max-height:100px;"><button data-id="${id}" class="remove-file">×</button></div>`
            : `<div id="${id}" class="file-preview"><span>${file.name}</span><button data-id="${id}" class="remove-file">×</button></div>`;
        this.$preview.append(tpl);
    }

    removeFile(id) { window.uploadedFiles = window.uploadedFiles.filter(f => f.id !== id); $(`#${id}`).remove(); }

    async handleSend() {
        const msg = this.$msg.val().trim();
        if (!msg && window.uploadedFiles.length === 0) return;
        this.ui.addChat(msg, 'user');
        this.$msg.val('').focus();
        // start status bubble
        this.ui.startStatus();
        await this.service.sendMessage(msg);
        // stop status bubble
        this.ui.clearStatus();
    }
}