<?php
namespace boru\boruaiweb\Http;

class Request
{
    /** @var array */
    private $query;
    /** @var array */
    private $post;
    /** @var array */
    private $server;
    /** @var array */
    private $cookies;
    /** @var array */
    private $files;
    /** @var array */
    private $headers;
    /** @var string|null */
    private $rawBody;
    /** @var array|null */
    private $jsonBody;

    /**
     * @param array $query
     * @param array $post
     * @param array $server
     * @param array $cookies
     * @param array $files
     * @param array $headers
     * @param string|null $rawBody
     * @param array|null $jsonBody
     */
    public function __construct(
        array $query,
        array $post,
        array $server,
        array $cookies,
        array $files,
        array $headers,
        $rawBody = null,
        $jsonBody = null
    ) {
        $this->query   = $query;
        $this->post    = $post;
        $this->server  = $server;
        $this->cookies = $cookies;
        $this->files   = $files;
        $this->headers = $headers;
        $this->rawBody = $rawBody;
        $this->jsonBody = $jsonBody;
    }

    /**
     * Create from PHP globals.
     * Supports JSON bodies (application/json) by decoding php://input and merging into params.
     *
     * @return Request
     */
    public static function fromGlobals()
    {
        $server = isset($_SERVER) ? $_SERVER : array();
        $headers = self::extractHeaders($server);

        $raw = null;
        if (function_exists('file_get_contents')) {
            $raw = @file_get_contents('php://input');
            if ($raw === false) $raw = null;
        }

        $get  = isset($_GET) ? $_GET : array();
        $post = isset($_POST) ? $_POST : array();

        // Parse JSON body if present
        $jsonBody = null;
        $contentType = isset($headers['content-type']) ? (string)$headers['content-type'] : '';

        if ($raw !== null) {
            $trim = trim($raw);
            if ($trim !== '' && self::isJsonContentType($contentType)) {
                $decoded = json_decode($trim, true);
                if (is_array($decoded)) {
                    $jsonBody = $decoded;

                    // Merge JSON keys into POST params when POST doesn't already contain them.
                    // This allows param() to work transparently for JSON requests.
                    foreach ($decoded as $k => $v) {
                        if (!array_key_exists($k, $post)) {
                            $post[$k] = $v;
                        }
                    }
                }
            }
        }

        return new self(
            $get,
            $post,
            $server,
            isset($_COOKIE) ? $_COOKIE : array(),
            isset($_FILES) ? $_FILES : array(),
            $headers,
            $raw,
            $jsonBody
        );
    }

    /**
     * Prefer POST over GET.
     * @param string $key
     * @param mixed $default
     * @return mixed
     */
    public function param($key, $default = null)
    {
        if (array_key_exists($key, $this->post)) return $this->post[$key];
        if (array_key_exists($key, $this->query)) return $this->query[$key];
        return $default;
    }

    /**
     * Combined params (POST overrides GET).
     * @return array
     */
    public function all()
    {
        $all = $this->query;
        foreach ($this->post as $k => $v) $all[$k] = $v;
        return $all;
    }

    /** @return array */
    public function query() { return $this->query; }
    /** @return array */
    public function post() { return $this->post; }
    /** @return array */
    public function server() { return $this->server; }
    /** @return array */
    public function cookies() { return $this->cookies; }
    /** @return array */
    public function files() { return $this->files; }

    /** @return string */
    public function method()
    {
        $m = isset($this->server['REQUEST_METHOD']) ? $this->server['REQUEST_METHOD'] : 'GET';
        return strtoupper((string)$m);
    }

    /**
     * @param string $name
     * @param mixed $default
     * @return mixed
     */
    public function header($name, $default = null)
    {
        $k = strtolower($name);
        return isset($this->headers[$k]) ? $this->headers[$k] : $default;
    }

    /** @return array lowercase header map */
    public function headers() { return $this->headers; }

    /** @return string|null */
    public function rawBody() { return $this->rawBody; }

    /**
     * Parsed JSON body (if Content-Type: application/json and decode succeeded)
     * @return array|null
     */
    public function json()
    {
        return $this->jsonBody;
    }

    /**
     * @param string $contentType
     * @return bool
     */
    private static function isJsonContentType($contentType)
    {
        $contentType = strtolower((string)$contentType);
        // handles: application/json, application/json; charset=utf-8, etc.
        return (strpos($contentType, 'application/json') !== false) ||
               (strpos($contentType, '+json') !== false);
    }

    /**
     * Extract headers from $_SERVER in a PHP-agnostic way.
     * @param array $server
     * @return array
     */
    private static function extractHeaders(array $server)
    {
        $headers = array();

        if (function_exists('getallheaders')) {
            $h = @getallheaders();
            if (is_array($h)) {
                foreach ($h as $k => $v) {
                    $headers[strtolower($k)] = $v;
                }
                return $headers;
            }
        }

        foreach ($server as $key => $value) {
            if (strpos($key, 'HTTP_') === 0) {
                $name = strtolower(str_replace('_', '-', substr($key, 5)));
                $headers[$name] = $value;
            } elseif ($key === 'CONTENT_TYPE') {
                $headers['content-type'] = $value;
            } elseif ($key === 'CONTENT_LENGTH') {
                $headers['content-length'] = $value;
            } elseif ($key === 'CONTENT_MD5') {
                $headers['content-md5'] = $value;
            }
        }

        return $headers;
    }
}
