<?php
namespace boru\boruaiweb\Providers\BoruAI\Chat;

use boru\boruaiweb\Contracts\ChatServiceInterface;
use boru\boruaiweb\Contracts\SettingsInterface;
use boru\boruaiweb\Domain\UserIdentity;

class BoruAiChatService implements ChatServiceInterface
{
    /** @var string */
    private $defaultModel;

    /** @var SettingsInterface|null */
    private $settings;

    /** @var string */
    private $settingsPrefix;

    /**
     * @param SettingsInterface|null $settings
     * @param string $settingsPrefix
     */
    public function __construct(SettingsInterface $settings = null, $settingsPrefix = 'boruaiweb.boruai.chat.')
    {
        $this->settings = $settings;
        $this->settingsPrefix = (string)$settingsPrefix;
        $this->defaultModel = (string)$this->get('default_model', 'gpt-4.1');
    }

    private function get($key, $default = null)
    {
        if (!$this->settings) return $default;
        return $this->settings->get($this->settingsPrefix . $key, $default);
    }

    /**
     * @param UserIdentity $user
     * @param string $reference
     * @param string $message
     * @param array $options
     * @param callable $emit function(array $event): void
     * @return void
     */
    public function streamChat(UserIdentity $user, $reference, $message, array $options, $emit)
    {
        if (!class_exists('\\boru\\boruai\\Models\\Response')) {
            $emit(array('type' => 'error', 'error' => 'BoruAI Response model not available'));
            return;
        }

        $reference = (string)$reference;
        $message = (string)$message;

        $perUserReference = isset($options['perUserReference']) ? (bool)$options['perUserReference'] : true;

        if ($perUserReference) {
            $response = \boru\boruai\Models\Response::withReference($reference, $user->id());
        } else {
            $response = \boru\boruai\Models\Response::withReference($reference);
        }

        // Model
        $model = isset($options['model']) && $options['model'] ? (string)$options['model'] : $this->defaultModel;
        if (method_exists($response, 'model')) {
            $response->model($model);
        }

        // Instructions
        $instructions = isset($options['instructions']) ? $options['instructions'] : null;
        if (is_string($instructions) && $instructions !== '' && method_exists($response, 'instructions')) {
            $response->instructions($instructions);
        }

        // If no message, legacy behavior was: use instructions as message (if provided)
        if ($message === '') {
            if (is_string($instructions) && $instructions !== '') {
                $message = $instructions;
                if (method_exists($response, 'instructions')) {
                    // treat as user message only
                    $response->instructions(null);
                }
            } else {
                $emit(array('type' => 'error', 'error' => 'No message provided'));
                return;
            }
        }

        // Tools
        $tools = isset($options['tools']) ? $options['tools'] : null;
        if (is_array($tools)) {
            foreach ($tools as $tool) {
                $loaded = $this->normalizeTool($tool);
                if ($loaded) {
                    if (method_exists($response, 'addTool')) {
                        $response->addTool($loaded);
                    }
                }
            }
        }

        // Add message + stream
        if (method_exists($response, 'addMessage')) {
            $response->addMessage($message);
        } else {
            // extremely defensive fallback
            $emit(array('type' => 'error', 'error' => 'Response model missing addMessage()'));
            return;
        }

        // Stream callback from BoruAI
        $response->create(null, function ($streamEvent) use ($emit) {
            $array = is_object($streamEvent) && method_exists($streamEvent, 'toArray')
                ? $streamEvent->toArray()
                : (array)$streamEvent;

            // Normalize: expose response_id + (optional) output_as_string on completion
            if (isset($array['response']) && is_array($array['response'])) {
                $type = isset($array['type']) ? (string)$array['type'] : '';

                // legacy typo existed in upstream UI ("reponse.complete") — support common variants
                $isComplete =
                    ($type === 'response.complete') ||
                    ($type === 'response.completed') ||
                    ($type === 'response.complete') ||
                    ($type === 'reponse.complete') ||
                    ($type === 'response.completed') ||
                    ($type === 'response.completed'); // harmless duplication

                if ($isComplete && class_exists('\\boru\\boruai\\Openai\\Response\\ResponseObject')) {
                    try {
                        $obj = new \boru\boruai\Openai\Response\ResponseObject($array['response']);
                        if (method_exists($obj, 'getResult')) {
                            $array['output_as_string'] = $obj->getResult(true);
                        }
                    } catch (\Exception $e) {
                        $array['output_as_string_error'] = $e->getMessage();
                    }
                }

                if (isset($array['response']['id'])) {
                    $array['response_id'] = $array['response']['id'];
                }

                unset($array['response']);
            }

            $emit($array);
        });
    }

    /**
     * @param mixed $tool
     * @return object|null
     */
    private function normalizeTool($tool)
    {
        // already an object tool
        if (is_object($tool)) {
            return $tool;
        }

        // string tool name => BoruAI::loadTool()
        if (is_string($tool) && $tool !== '' && class_exists('\\boru\\boruai\\BoruAI')) {
            if (method_exists('\\boru\\boruai\\BoruAI', 'loadTool')) {
                $obj = \boru\boruai\BoruAI::loadTool($tool);
                if (is_object($obj)) return $obj;
            }
        }

        return null;
    }
}
