<?php
namespace boru\boruaiweb;

use boru\boruaiweb\Contracts\AuthProviderInterface;
use boru\boruaiweb\Contracts\ChatServiceInterface;
use boru\boruaiweb\Contracts\TemplateRepositoryInterface;
use boru\boruaiweb\Contracts\ResponseRepositoryInterface;
use boru\boruaiweb\Contracts\HistoryRepositoryInterface;
use boru\boruaiweb\Contracts\ReferenceRepositoryInterface;
use boru\boruaiweb\Contracts\ToolCatalogInterface;
use boru\boruaiweb\Contracts\SettingsInterface;

class WebConfig implements SettingsInterface
{
    /** @var string|null */
    public $userTemplateDir;
    /** @var string|null */
    public $userAssetDir;
    /** @var string */
    public $assetUrlPrefix = '/assets/boruaiweb';
    /** @var string */
    public $pageParam = 'page';
    /** @var string */
    public $actionParam = 'action';
    /** @var string */
    public $defaultPage = 'dashboard';

    /** @var AuthProviderInterface|null */
    public $auth;
    /** @var ChatServiceInterface|null */
    public $chat;
    /** @var TemplateRepositoryInterface|null */
    public $templates;
    /** @var ResponseRepositoryInterface|null */
    public $responses;
    /** @var HistoryRepositoryInterface|null */
    public $history;
    /** @var ReferenceRepositoryInterface|null */
    public $references;
    /** @var ToolCatalogInterface|null */
    public $tools;

    /** @var bool */
    public $perUserReference = true;

    // Packaged defaults (auto)
    public $packageResourceDir;

    /** @var string|null */
    public $publicDir;

    /** @var array */
    private $settings = array();

    /**
     * SettingsInterface
     * @param string $key
     * @param mixed $default
     * @return mixed
     */
    public function get($key, $default = null)
    {
        $k = (string)$key;
        return array_key_exists($k, $this->settings) ? $this->settings[$k] : $default;
    }

    /**
     * SettingsInterface
     * @param string $key
     * @param mixed $value
     * @return $this
     */
    public function set($key, $value)
    {
        $this->settings[(string)$key] = $value;
        return $this;
    }

    /**
     * SettingsInterface
     * @return array
     */
    public function all()
    {
        return $this->settings;
    }

    public static function createDefault()
    {
        $c = new self();

        // Infer public dir from entry script
        if (isset($_SERVER['SCRIPT_FILENAME'])) {
            $c->publicDir = dirname($_SERVER['SCRIPT_FILENAME']);
        } else {
            $c->publicDir = getcwd();
        }

        // Infer mount path from URL (e.g. /boruaiweb/index.php => /boruaiweb/)
        $scriptName = isset($_SERVER['SCRIPT_NAME']) ? (string)$_SERVER['SCRIPT_NAME'] : '';
        $mount = '/';
        if ($scriptName !== '') {
            $mount = rtrim(str_replace('\\', '/', dirname($scriptName)), '/');
            if ($mount === '' || $mount === '.') $mount = '/';
            $mount = rtrim($mount, '/') . '/';
        }

        // IMPORTANT: this is the "public root" for the app, not the assets directory itself
        $c->assetUrlPrefix = $mount;

        return $c;
    }


    public function withTemplateDir($dir) { $this->userTemplateDir = $dir; return $this; }
    public function withAssetDir($dir) { $this->userAssetDir = $dir; return $this; }
    public function withAssetUrlPrefix($prefix) { $this->assetUrlPrefix = $prefix; return $this; }

    public function withAuth(AuthProviderInterface $auth) { $this->auth = $auth; return $this; }
    public function withChat(ChatServiceInterface $chat) { $this->chat = $chat; return $this; }
    public function withTemplates(TemplateRepositoryInterface $repo) { $this->templates = $repo; return $this; }
    public function withResponses(ResponseRepositoryInterface $repo) { $this->responses = $repo; return $this; }
    public function withHistory(HistoryRepositoryInterface $repo) { $this->history = $repo; return $this; }
    public function withReferences(ReferenceRepositoryInterface $repo) { $this->references = $repo; return $this; }
    public function withTools(ToolCatalogInterface $tools) { $this->tools = $tools; return $this; }
    /**
     * Optional helper if you want to set many at once.
     * @param array $settings
     * @return $this
     */
    public function withSettings(array $settings)
    {
        foreach ($settings as $k => $v) {
            $this->settings[(string)$k] = $v;
        }
        return $this;
    }
    public function withPublicDir($dir)
    {
        $this->publicDir = rtrim($dir, DIRECTORY_SEPARATOR);
        return $this;
    }
}
