<?php
namespace boru\boruaiweb\Assets;

class AssetManager
{
    /** @var string */
    private $mountUrl; // e.g. "/boruaiweb/"
    /** @var string */
    private $publicDir; // filesystem dir containing "assets/"
    /** @var string */
    private $assetsUrlRoot; // mountUrl + "assets/"

    /**
     * @param string $mountUrl     URL prefix for the app root (ends with '/'), e.g. "/boruaiweb/"
     * @param string $publicDir    filesystem directory containing "assets" folder (usually where index.php lives)
     */
    public function __construct($mountUrl, $publicDir)
    {
        $mountUrl = (string)$mountUrl;
        if ($mountUrl === '') $mountUrl = '/';
        $mountUrl = rtrim($mountUrl, '/') . '/';

        $this->mountUrl = $mountUrl;
        $this->publicDir = rtrim((string)$publicDir, DIRECTORY_SEPARATOR);
        $this->assetsUrlRoot = $this->mountUrl . 'assets/';
    }

    /**
     * Resolve a JS file URL using custom-first fallback.
     * @param string $file e.g. "ChatService.js"
     * @return string
     */
    public function js($file)
    {
        return $this->resolve('js', $file);
    }

    /**
     * Resolve a CSS file URL using custom-first fallback.
     * @param string $file e.g. "app.css" or "fa/css/fontawesome.min.css"
     * @return string
     */
    public function css($file)
    {
        return $this->resolve('css', $file);
    }

    /**
     * A generic resolver for other static files if needed.
     * @param string $type "css"|"js"|...
     * @param string $relative
     * @return string
     */
    public function file($type, $relative)
    {
        return $this->resolve($type, $relative);
    }

    /**
     * Compatibility helper: returns mountUrl for non-asset relative references.
     * @return string
     */
    public function mount()
    {
        return $this->mountUrl;
    }

    /**
     * @param string $type
     * @param string $relative
     * @return string
     */
    private function resolve($type, $relative)
    {
        $type = trim((string)$type, '/\\');
        $relative = ltrim((string)$relative, '/\\');

        // filesystem checks
        $customPath = $this->publicDir . DIRECTORY_SEPARATOR . 'assets' . DIRECTORY_SEPARATOR . 'custom'
            . DIRECTORY_SEPARATOR . $type . DIRECTORY_SEPARATOR . str_replace('/', DIRECTORY_SEPARATOR, $relative);

        if (is_file($customPath)) {
            return $this->assetsUrlRoot . 'custom/' . $type . '/' . $this->urlify($relative);
        }

        // default fallback
        return $this->assetsUrlRoot . 'default/' . $type . '/' . $this->urlify($relative);
    }

    /**
     * @param string $path
     * @return string
     */
    private function urlify($path)
    {
        // ensure forward slashes for URLs
        return str_replace('\\', '/', $path);
    }
}
