<?php
namespace boru\boruaiweb\Publishing;

use boru\boruaiweb\Http\HttpException;
use boru\boruaiweb\WebConfig;

class DefaultPublisher
{
    /** @var WebConfig */
    private $config;

    public function __construct(WebConfig $config)
    {
        $this->config = $config;
    }

    /**
     * Check if defaults appear published (basic sentinel checks).
     * @return bool
     */
    public function isPublished()
    {
        $publicDir = $this->config->publicDir ? $this->config->publicDir : getcwd();

        $assetDefaultCss = $publicDir . DIRECTORY_SEPARATOR . 'assets' . DIRECTORY_SEPARATOR . 'default' . DIRECTORY_SEPARATOR . 'css';
        $assetDefaultJs  = $publicDir . DIRECTORY_SEPARATOR . 'assets' . DIRECTORY_SEPARATOR . 'default' . DIRECTORY_SEPARATOR . 'js';

        // If either default css/js missing, treat as not published.
        if (!is_dir($assetDefaultCss) || !is_dir($assetDefaultJs)) return false;

        // Template default directory sentinel
        if ($this->config->userTemplateDir) {
            $tplDefault = rtrim($this->config->userTemplateDir, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . 'default';
            if (!is_dir($tplDefault)) return false;
        }

        return true;
    }

    /**
     * Publish assets/templates into new structure.
     *
     * assets/default/*  (copied from resources/assets/*)
     * assets/custom/*   (created empty)
     *
     * templates/default/* (copied from resources/templates/*)
     * templates/custom/*  (created empty)
     *
     * @param bool $overwrite
     * @return void
     */
    public function publish($overwrite = false)
    {
        try {
            $pkgRoot = dirname(dirname(__DIR__)); // src -> package root
            $resRoot = $pkgRoot . DIRECTORY_SEPARATOR . 'resources';

            $publicDir = $this->config->publicDir ? $this->config->publicDir : getcwd();
            $assetsRoot = $publicDir . DIRECTORY_SEPARATOR . 'assets';

            // Ensure directory structure
            $this->mkdir($assetsRoot . DIRECTORY_SEPARATOR . 'default' . DIRECTORY_SEPARATOR . 'css');
            $this->mkdir($assetsRoot . DIRECTORY_SEPARATOR . 'default' . DIRECTORY_SEPARATOR . 'js');
            $this->mkdir($assetsRoot . DIRECTORY_SEPARATOR . 'custom'  . DIRECTORY_SEPARATOR . 'css');
            $this->mkdir($assetsRoot . DIRECTORY_SEPARATOR . 'custom'  . DIRECTORY_SEPARATOR . 'js');

            // Copy packaged assets into assets/default (preserve subdirs like fa/)
            $assetSrc = $resRoot . DIRECTORY_SEPARATOR . 'assets';
            $assetDst = $assetsRoot . DIRECTORY_SEPARATOR . 'default';
            if (is_dir($assetSrc)) {
                Publisher::copyDir($assetSrc, $assetDst, $overwrite);
            }

            // Templates
            if ($this->config->userTemplateDir) {
                $tplRoot = rtrim($this->config->userTemplateDir, DIRECTORY_SEPARATOR);
                $this->mkdir($tplRoot . DIRECTORY_SEPARATOR . 'default');
                $this->mkdir($tplRoot . DIRECTORY_SEPARATOR . 'custom');

                $tplSrc = $resRoot . DIRECTORY_SEPARATOR . 'templates';
                $tplDst = $tplRoot . DIRECTORY_SEPARATOR . 'default';
                if (is_dir($tplSrc)) {
                    Publisher::copyDir($tplSrc, $tplDst, $overwrite);
                }
            }
        } catch (\Exception $e) {
            throw HttpException::serviceUnavailable(
                'Exception during Publish.. Static files could not be initialized.',
                "Publish failed:\n" . $e->getMessage(),
                $e
            );
        }
    }

    private function mkdir($dir)
    {
        if (!is_dir($dir)) {
            if (!mkdir($dir, 0777, true) && !is_dir($dir)) {
                throw new \RuntimeException("Failed to create directory: " . $dir);
            }
        }
    }
}
