<?php
namespace boru\boruelastic\conn;

use boru\boruelastic\Elastic;
use boru\dhttp\Client;
use GuzzleHttp\Promise\PromiseInterface;

class Connector {
    private $endpoint;
    private $auth;
    private $async = false;
    public function __construct($endpoint,$options=[]) {
        $this->endpoint($endpoint);
        if(isset($options["auth"])) {
            $this->auth($options["auth"]);
            unset($options["auth"]);
        }
        if(isset($options["async"])) {
            $this->async($options["async"]);
            unset($options["async"]);
        }
        Elastic::setClient($this);
    }
    public function endpoint($endpoint=null) {
        if(is_null($endpoint)) return $this->endpoint;
        $this->endpoint = $endpoint;
    }
    public function auth($auth=null) {
        if(is_null($auth)) return $this->auth;
        if(is_array($auth)) {
            $u = $p = "";
            if(isset($auth["user"])) {
                $u = $auth["user"];
            } elseif(isset($auth["username"])) {
                $u = $auth["username"];
            }
            if(isset($auth["pass"])) {
                $p = $auth["pass"];
            } elseif(isset($auth["password"])) {
                $p = $auth["password"];
            }
            if($u && $p) {
                $auth = $u.":".$p;
            } else {
                $auth = implode(":",$auth);
            }
        }
        $this->auth = $auth;
    }
    public function async($async=null) {
        if(is_null($async)) return $this->async;
        $this->async = $async;
    }

    public function getRequest($path,$data=[],$async=null) {
        return $this->request("GET",$path,$data,$async);
    }
    public function postRequest($path,$data=[],$async=null) {
        return $this->request("POST",$path,$data,$async);
    }
    public function putRequest($path,$data=[],$async=null) {
        return $this->request("PUT",$path,$data,$async);
    }
    public function deleteRequest($path,$data=[],$async=null) {
        return $this->request("DELETE",$path,$data,$async);
    }
    public function headRequest($path,$data=[],$async=null) {
        return $this->request("HEAD",$path,$data,$async);
    }

    public function request($method="GET",$path,$data=[],$async=null) {
        if(is_null($async)) {
            $async = $this->async();
        }
        $url = $this->pathToUrl($path);
        $options = $this->buildRequestOptions($data);
        return $this->rawRequest($method,$url,$options,$async);
    }
    public function rawRequest($method="GET",$url,$options=[],$async=null) {
        if(strtoupper($method) == "GET") {
            $response = Client::get($url,$options,$async);
        } elseif(strtoupper($method) == "POST") {
            $response = Client::post($url,$options,$async);
        } elseif(strtoupper($method) == "PUT") {
            $response = Client::put($url,$options,$async);
        } elseif(strtoupper($method) == "DELETE") {
            $response = Client::delete($url,$options,$async);
        } elseif(strtoupper($method) == "HEAD") {
            $response = Client::head($url,$options,$async);
        } else {
            throw new \Exception("Method not supported");
        }
        $this->throwIfError($response);
        if($this->isPromise($response)) {
            return $response;
        }
        return $response->asArray();
    }

    //helper functions
    public function _index($index,$data=[],$async=null) {
        return $this->putRequest($index,$data,$async);
    }
    public function _template($templateName,$data=[],$async=null) {
        $path = "_template/".$templateName;
        return $this->putRequest($path,$data,$async);
    }
    public function _update($index,$id,$data=[],$async=null) {
        $path = $index."/_doc/".$id;
        return $this->postRequest($path,$data,$async);
    }
    public function _insert($index,$id,$data=[],$async=null) {
        $path = $index."/_doc/".$id;
        return $this->putRequest($path,$data,$async);
    }
    public function _bulk($data=[],$onlyErrorReturn=false,$async=null) {
        $path = "/_bulk";
        if($onlyErrorReturn) {
            $path .= "?filter_path=items.*.error";
        }
        $bulkData = "";
        foreach($data as $item) {
            $bulkData .= json_encode($item)."\n";
        }
        if(is_null($async)) {
            $async = $this->async();
        }
        $url = $this->pathToUrl("_bulk");
        $options = $this->buildRequestOptions();
        $options["body"] = $bulkData;
        return $this->rawRequest("POST",$url,$options,$async);
    }
    private function buildRequestOptions($data=[]) {
        $options = [];
        if(!empty($data)) {
            if(is_array($data)) {
                $options["json"] = json_encode($data);
            } else {
                $options["json"] = $data;
            }
        }
        $options["headers"] = ["Content-Type"=>"application/json"];
        if(!is_null($this->auth)) {
            $t = explode(":",$this->auth);
            if(count($t) == 2) {
                $options["auth"] = $t; //basic auth
            } else {
                $options["auth"] = $this->auth; //token
            }
        }
        return $options;
    }
    private function isPromise($data) {
        return $data instanceof PromiseInterface;
    }
    private function throwIfError($data,$throwIfFalse=true) {
        if($data instanceof \Exception) throw $data;
        if($data === false && $throwIfFalse) {
            throw new \Exception("Request failed");
            return;
        }
        return $data;
    }
    private function pathToUrl($path) {
        if(substr($path,0,4) != "http") {
            $url = $this->endpoint();
            if(substr($url,-1) != "/") $url .= "/";
            if(substr($path,0,1) == "/") $path = substr($path,1);
            $url .= $path;
        } else {
            $url = $path;
        }
        return $url;
    }
}