<?php
namespace boru\boruelastic\conn;

use boru\boruelastic\Elastic;
use boru\dhttp\Client;
use boru\dhttp\HttpClient;
use Exception;
use GuzzleHttp\Promise\PromiseInterface;

class Connector {
    private $endpoint;
    private $auth;
    private $async = false;
    private $httpClient;
    public function __construct($endpoint,$options=[],$client=null) {
        if(isset($options["client"])) {
            $this->httpClient = $options["client"];
            unset($options["client"]);
        }
        if(!is_null($client)) {
            $this->httpClient = $client;
        }
        if(is_null($this->httpClient)) {
            $this->httpClient = new HttpClient($options);
        }
        $this->endpoint($endpoint);
        if(isset($options["auth"])) {
            $this->auth($options["auth"]);
            unset($options["auth"]);
        }
        if(isset($options["async"])) {
            $this->async($options["async"]);
            unset($options["async"]);
        }
        Elastic::setClient($this);
    }

    /**
     * Set or get the endpoint
     * @param string|null $endpoint
     * @return string
     */
    public function endpoint($endpoint=null) {
        if(!is_null($endpoint)) {
            $this->endpoint = $endpoint;
        }
        return $this->endpoint;
    }

    /**
     * Set or get the auth
     * @param string|array|null $auth
     * @return string
     */
    public function auth($auth=null) {
        if(is_array($auth)) {
            $u = $p = "";
            if(isset($auth["user"])) {
                $u = $auth["user"];
            } elseif(isset($auth["username"])) {
                $u = $auth["username"];
            }
            if(isset($auth["pass"])) {
                $p = $auth["pass"];
            } elseif(isset($auth["password"])) {
                $p = $auth["password"];
            }
            if($u && $p) {
                $auth = $u.":".$p;
            } else {
                $auth = implode(":",$auth);
            }
        }
        if(!is_null($auth)) {
            $this->auth = $auth;
        }
        return $this->auth;
    }

    /**
     * Set or get the async
     * @param bool|null $async
     * @return bool
     */
    public function async($async=null) {
        if(!is_null($async)) {
            $this->async = $async;
        }
        return $this->async;
    }

    /**
     * Send a GET request
     * @param string $index
     * @return bool|\React\Promise\PromiseInterface
     * @throws \Exception
     */
    public function getRequest($path,$data=[],$async=null) {
        return $this->request("GET",$path,$data,$async);
    }
    /**
     * Send a POST request
     * @param string $index
     * @return bool|\React\Promise\PromiseInterface
     * @throws \Exception
     */
    public function postRequest($path,$data=[],$async=null) {
        return $this->request("POST",$path,$data,$async);
    }
    /**
     * Send a PUT request
     * @param string $index
     * @return bool|\React\Promise\PromiseInterface
     * @throws \Exception
     */
    public function putRequest($path,$data=[],$async=null) {
        return $this->request("PUT",$path,$data,$async);
    }
    /**
     * Send a DELETE request
     * @param string $index
     * @return bool|\React\Promise\PromiseInterface
     * @throws \Exception
     */
    public function deleteRequest($path,$data=[],$async=null) {
        return $this->request("DELETE",$path,$data,$async);
    }
    /**
     * Send a HEAD request
     * @param string $index
     * @return bool|\React\Promise\PromiseInterface
     * @throws \Exception
     */
    public function headRequest($path,$data=[],$async=null) {
        return $this->request("HEAD",$path,$data,$async);
    }

    /**
     * Send a request
     * @param string $method
     * @param string $path
     * @param array $data
     * @param bool|null $async
     * @return bool|\React\Promise\PromiseInterface
     * @throws \Exception
     */
    public function request($method="GET",$path,$data=[],$async=null) {
        if(is_null($async)) {
            $async = $this->async();
        }
        $url = $this->pathToUrl($path);
        $options = $this->buildRequestOptions($data);
        return $this->rawRequest($method,$url,$options,$async);
    }

    /**
     * Send a raw request
     * @param string $method 
     * @param mixed $url 
     * @param array $options 
     * @param mixed $async 
     * @return mixed 
     * @throws Exception 
     */
    public function rawRequest($method="GET",$url,$options=[],$async=null) {
        if(strtoupper($method) == "GET") {
            $response = $this->httpClient->get($url,$options,$async);
        } elseif(strtoupper($method) == "POST") {
            $response = $this->httpClient->post($url,$options,$async);
        } elseif(strtoupper($method) == "PUT") {
            $response = $this->httpClient->put($url,$options,$async);
        } elseif(strtoupper($method) == "DELETE") {
            $response = $this->httpClient->delete($url,$options,$async);
        } elseif(strtoupper($method) == "HEAD") {
            $response = $this->httpClient->head($url,$options,$async);
        } else {
            throw new \Exception("Method not supported");
        }
        $this->throwIfError($response);
        if($this->isPromise($response)) {
            return $response;
        }
        return $response->asArray();
    }

    //helper functions
    public function _index($index,$data=[],$async=null) {
        return $this->putRequest($index,$data,$async);
    }
    public function _template($templateName,$data=[],$async=null) {
        $path = "_template/".$templateName;
        return $this->putRequest($path,$data,$async);
    }
    public function _update($index,$id,$data=[],$async=null) {
        $path = $index."/_doc/".$id;
        return $this->postRequest($path,$data,$async);
    }
    public function _insert($index,$id,$data=[],$async=null) {
        $path = $index."/_doc/".$id;
        return $this->putRequest($path,$data,$async);
    }
    public function _bulk($data=[],$onlyErrorReturn=false,$async=null) {
        $path = "/_bulk";
        if($onlyErrorReturn) {
            $path .= "?filter_path=items.*.error";
        }
        $bulkData = "";
        foreach($data as $item) {
            $bulkData .= json_encode($item)."\n";
        }
        if(is_null($async)) {
            $async = $this->async();
        }
        $url = $this->pathToUrl("_bulk");
        $options = $this->buildRequestOptions();
        $options["body"] = $bulkData;
        return $this->rawRequest("POST",$url,$options,$async);
    }
    private function buildRequestOptions($data=[]) {
        $options = [];
        if(!empty($data)) {
            if(is_array($data)) {
                $options["json"] = json_encode($data,JSON_PRETTY_PRINT|JSON_UNESCAPED_SLASHES);
            } else {
                $options["json"] = $data;
            }
        }
        $options["headers"] = ["Content-Type"=>"application/json"];
        if(!is_null($this->auth)) {
            $t = explode(":",$this->auth);
            if(count($t) == 2) {
                $options["auth"] = $t; //basic auth
            } else {
                $options["auth"] = $this->auth; //token
            }
        }
        return $options;
    }
    private function isPromise($data) {
        return $data instanceof PromiseInterface;
    }
    private function throwIfError($data,$throwIfFalse=true) {
        if($data instanceof \Exception) throw $data;
        if($data === false && $throwIfFalse) {
            throw new \Exception("Request failed");
            return;
        }
        return $data;
    }
    private function pathToUrl($path) {
        if(substr($path,0,4) != "http") {
            $url = $this->endpoint();
            if(substr($url,-1) != "/") $url .= "/";
            if(substr($path,0,1) == "/") $path = substr($path,1);
            $url .= $path;
        } else {
            $url = $path;
        }
        return $url;
    }
}