<?php
namespace boru\borumcp\Authenticators;

use boru\borumcp\Core\JWT;
use boru\borumcp\Interfaces\AuthenticatorInterface;

/**
 * Verifies HS256 or RS256 tokens; checks exp/nbf/iat; optional iss/aud.
 */
final class JWTAuthenticator implements AuthenticatorInterface
{
    private $alg;            // 'HS256' | 'RS256'
    private $verifyKey;      // HS: secret; RS: public key (PEM)
    private $requireIss;     // string|null
    private $requireAud;     // string|string[]|null
    private $leeway;         // seconds of clock skew

    private $ttl = 300; // default lifetime for minted tokens
    private $kid = null; // default kid for minted tokens

    /** @var JWTSigner|null */
    private $signer = null; // JwtSigner|null

    public function __construct($alg, $verifyKey, $requireIss = null, $requireAud = null, $leeway = 60, $ttl = 300, $kid = null) {
        $this->alg = strtoupper($alg);
        $this->verifyKey = $verifyKey;
        $this->requireIss = $requireIss;
        $this->requireAud = $requireAud;
        $this->leeway = (int)$leeway;
        $this->ttl = (int)$ttl;
        $this->kid = $kid;
    }

    public function verify($token) {
        if (!is_string($token) || $token === '') return null;
        list($ok, $claims, $err,) = JWT::decodeAndVerify($token, $this->alg, $this->verifyKey);
        if (!$ok) return null;

        $now = time();
        if (isset($claims['nbf']) && $now + $this->leeway < (int)$claims['nbf']) return null;
        if (isset($claims['iat']) && $now + $this->leeway < (int)$claims['iat']) return null;
        if (isset($claims['exp']) && $now - $this->leeway >= (int)$claims['exp']) return null;

        if ($this->requireIss !== null) {
            if (!isset($claims['iss']) || $claims['iss'] !== $this->requireIss) return null;
        }
        if ($this->requireAud !== null) {
            $aud = isset($claims['aud']) ? $claims['aud'] : null;
            if (is_array($this->requireAud)) {
                if (!in_array($aud, $this->requireAud, true)) return null;
            } else {
                if ($aud !== $this->requireAud) return null;
            }
        }

        // Map to principal (customize as you like)
        $principal = array(
            'sub'   => isset($claims['sub']) ? $claims['sub'] : 'unknown',
            'roles' => isset($claims['roles']) && is_array($claims['roles']) ? $claims['roles'] : array('default'),
            'claims'=> $claims
        );
        return $principal;
    }

    public function getSigner() {
        if ($this->signer !== null) return $this->signer;
        if ($this->alg !== 'HS256' && $this->alg !== 'RS256') return null;
        $this->signer = new \boru\borumcp\Core\JwtSigner($this->alg, $this->verifyKey, $this->kid, $this->requireIss, $this->requireAud, $this->ttl);
        return $this->signer;
    }

    public function getMinter($signKey=null) {
        if ($signKey === null) $signKey = $this->verifyKey;
        return new \boru\borumcp\Core\JwtMinter($this, $signKey, array('alg'=>$this->alg, 'kid'=>$this->kid, 'iss'=>$this->requireIss, 'aud'=>$this->requireAud, 'ttl'=>$this->ttl));
    }

    public function setIss($iss) { $this->requireIss = $iss; }
    public function setAud($aud) { $this->requireAud = $aud; }
    public function setLeeway($leeway) { $this->leeway = (int)$leeway; }
    public function setTtl($ttl) { $this->ttl = (int)$ttl; }
    public function setKid($kid) { $this->kid = $kid; }

    public function getAlg() { return $this->alg; }
    public function getRequiredIss() { return $this->requireIss; }
    public function getRequiredAud() { return $this->requireAud; }
    public function getLeeway() { return $this->leeway; }
    public function getTtl() { return $this->ttl; }
    public function getKid() { return $this->kid; }
}