<?php
namespace boru\borumcp\Core;

use boru\borumcp\Interfaces\AuthenticatorInterface;

/**
 * Per-connection session (handles buffering lines).
 */
final class McpSession {
    private $buffer='';

    /** @var McpDispatcher */
    private $dispatcher;

    /** @var AuthenticatorInterface|null */
    private $authenticator = null;

    /** @var bool */
    private $isAuthenticated = false;

    /** @var callable|null */
    private $closeFn = null;

    /** @var callable|null */
    private $writeFn = null;

    public function __construct(McpDispatcher $dispatcher, AuthenticatorInterface $authenticator = null, $closeFn = null){
        $this->dispatcher = $dispatcher;
        $this->authenticator = $authenticator;
        $this->closeFn = $closeFn; // function():void that terminates the connection
    }

    public function setWriteFn($fn){ $this->writeFn = $fn; }

    private function sendLine($jsonrpcObj){
        if (is_callable($this->writeFn)) {
            call_user_func($this->writeFn, $jsonrpcObj);
        }
    }

    public function onData($chunk){
        $this->buffer.=$chunk;
        $parts=explode("\n",$this->buffer);
        $this->buffer=array_pop($parts);
        foreach($parts as $line){
            $trim = trim($line);
            if ($trim === '') { continue; }

            if (!$this->isAuthenticated && $this->authenticator) {
                // Require the very first JSON-RPC to be initialize with params.authToken
                $msg = json_decode($trim, true);
                if (!is_array($msg) || !isset($msg['method'])) {
                    // ignore non-JSON until initialize arrives (or drop)
                    continue;
                }
                $id = isset($msg['id']) ? $msg['id'] : null;
                if ($msg['method'] !== 'initialize') {
                    // Must initialize first
                    if ($id !== null) {
                        $err = JsonRpc::error($id, 401, 'Unauthorized: initialize required before other methods');
                        $this->sendLine($err);
                    }
                    $this->closeConn();
                    return;
                }
                $params = isset($msg['params']) && is_array($msg['params']) ? $msg['params'] : array();
                $token  = isset($params['authToken']) ? $params['authToken'] : null;

                $principal = $this->authenticator->verify($token);
                if (!$principal) {
                    if ($id !== null) {
                        $err = JsonRpc::error($id, 401, 'Unauthorized: invalid token');
                        $this->sendLine($err);
                    }
                    $this->closeConn();
                    return;
                }

                // Success: store principal on dispatcher and mark authenticated
                if (method_exists($this->dispatcher, 'setPrincipal')) {
                    $this->dispatcher->setPrincipal($principal);
                }
                $this->isAuthenticated = true;

                // Fall through and deliver the same initialize line to dispatcher
            }

            $this->dispatcher->handleLine($line);
        }
    }

    private function closeConn(){
        if (is_callable($this->closeFn)) {
            call_user_func($this->closeFn);
        }
    }

    public function flushRemainder(){
        if($this->buffer!==''){ $this->dispatcher->handleLine($this->buffer); $this->buffer=''; }
    }
}