<?php
namespace boru\borumcp;

/**
 * HttpProxyController
 * -------------------
 * Class-based HTTP→TCP MCP proxy for Apache/PHP-FPM usage (no React server).
 *
 * Usage (Apache-exposed script):
 *   require 'vendor/autoload.php';
 *   $proxy = (new \boru\borumcp\HttpProxyController())
 *       ->upstreamHost('127.0.0.1')
 *       ->allowedPortRange(8000, 65000)
 *       ->internalPortHeader('internal-port')
 *       ->timeouts(3.0, 2.0, 4000, 4000, 8000);
 *   $proxy->handle(); // echoes JSON response & exits
 *
 * Endpoints:
 *   POST /tools/list
 *   POST /tools/call
 *   GET  /health
 *
 * Security notes:
 *   - Only connects to upstreamHost (default 127.0.0.1).
 *   - Validates internal-port is numeric and within allowed range.
 *   - Forwards bearer token to upstream MCP TCP server (which should verify JWT).
 */
class HttpProxyController
{
    /** Where ephemeral TCP servers listen (loopback by default) */
    private $upstreamHost = '127.0.0.1';

    /** Allowed upstream port range */
    private $minPort = 8000;
    private $maxPort = 65000;

    /** Header used to choose port */
    private $internalPortHeader = 'internal-port';

    /** Timeouts (seconds / milliseconds) */
    private $connectTimeout = 3.0;      // fsockopen connect timeout
    private $readTimeout    = 2.0;      // per fgets() timeout (stream_set_timeout)
    private $initTimeoutMs  = 4000;     // initialize round-trip deadline
    private $listTimeoutMs  = 4000;     // tools/list deadline
    private $callTimeoutMs  = 8000;     // tools/call deadline

    /** MCP protocol version forwarded to the TCP server on initialize */
    private $protocolVersion = '2025-06-18';

    /** Optional log file path; null = no logging */
    private $logFile = null;

    // -------------------- Constructor --------------------

    /**
     * @param string|null $logFile Optional path to write verbose logs.
     */
    public function __construct($logFile = null)
    {
        if ($logFile !== null) {
            $this->logFile = (string)$logFile;
        }
    }

    // -------------------- Fluent configuration --------------------

    public function upstreamHost($host)
    {
        $this->upstreamHost = (string)$host;
        return $this;
    }
    public function allowedPortRange($min, $max)
    {
        $this->minPort = (int)$min;
        $this->maxPort = (int)$max;
        return $this;
    }
    public function internalPortHeader($name)
    {
        $this->internalPortHeader = (string)$name;
        return $this;
    }
    public function timeouts($connectTimeoutSeconds = 3.0, $readTimeoutSeconds = 2.0, $initMs = 4000, $listMs = 4000, $callMs = 8000)
    {
        $this->connectTimeout = (float)$connectTimeoutSeconds;
        $this->readTimeout    = (float)$readTimeoutSeconds;
        $this->initTimeoutMs  = (int)$initMs;
        $this->listTimeoutMs  = (int)$listMs;
        $this->callTimeoutMs  = (int)$callMs;
        return $this;
    }
    public function protocolVersion($pv)
    {
        $this->protocolVersion = (string)$pv;
        return $this;
    }

    public function logFile($path)
    {
        $this->logFile = $path;
        return $this;
    }

    // -------------------- Main entry (Apache) ---------------------

    /**
     * Handle the current HTTP request using PHP superglobals.
     * Emits JSON response and exits.
     */
    public function handle()
    {
        // Always return JSON
        header('Content-Type: application/json');

        $path   = $this->requestPath();
        $method = $this->requestMethod();
        $headers = $this->headers();
        $bodyRaw = file_get_contents('php://input');

        $this->log("Incoming {$method} {$path}", [
            'headers' => $headers,
            'body'    => $bodyRaw,
            'remote_addr' => $_SERVER['REMOTE_ADDR'] ?? null
        ]);

        if ($method === 'GET' && $path === '/health') {
            $this->reply(200, array('ok' => true, 'status' => 'healthy'));
        }

        if ($method !== 'POST') {
            $this->error(405, 'Method Not Allowed');
        }
        if ($path !== '/tools/list' && $path !== '/tools/call') {
            $this->error(404, 'Not Found');
        }

        $portHeader = $this->headerValue($headers, $this->internalPortHeader);
        if ($portHeader === null || $portHeader === '') {
            $this->error(400, 'Missing required header: '.$this->internalPortHeader);
        }
        if (!ctype_digit($portHeader)) {
            $this->error(400, 'Invalid '.$this->internalPortHeader.' header (digits only)');
        }
        $port = (int)$portHeader;
        if ($port < $this->minPort || $port > $this->maxPort) {
            $this->error(403, 'Port not allowed', array('allowed_range' => $this->minPort.'-'.$this->maxPort));
        }

        $bearer = $this->parseBearer($headers);

        // Parse body only for tools/call
        $body = array();
        if ($path === '/tools/call' && $bodyRaw !== '' && $bodyRaw !== false) {
            $body = json_decode($bodyRaw, true);
            if (!is_array($body)) {
                $this->error(400, 'Expected JSON body');
            }
        }

        // Create one short-lived TCP session for this HTTP request
        $sock = $this->tcpConnect($this->upstreamHost, $port, $this->connectTimeout);
        if (!$sock) {
            $this->error(502, 'Upstream TCP connect failed', array('port' => $port));
        }

        stream_set_blocking($sock, true);
        stream_set_timeout($sock, $this->readTimeout);

        $initParams = array('protocolVersion' => $this->protocolVersion);
        if ($bearer !== null) { $initParams['authToken'] = $bearer; }
        $this->rpcSend($sock, 1, 'initialize', $initParams);
        $initResp = $this->rpcReadUntilId($sock, 1, $this->initTimeoutMs);

        $this->log("Init response", $initResp);

        if ($initResp === null || isset($initResp['error'])) {
            fclose($sock);
            $this->error(401, 'Upstream initialize failed', array('upstream_error' => isset($initResp['error']) ? $initResp['error'] : 'timeout'));
        }

        if ($path === '/tools/list') {
            $this->rpcSend($sock, 2, 'tools/list', new \stdClass());
            $resp = $this->rpcReadUntilId($sock, 2, $this->listTimeoutMs);
            fclose($sock);
            $this->log("tools/list response", $resp);

            if ($resp === null || isset($resp['error'])) {
                $this->error(502, 'Upstream tools/list error', array('upstream_error' => isset($resp['error']) ? $resp['error'] : 'timeout'));
            }
            $result = isset($resp['result']) && is_array($resp['result']) ? $resp['result'] : array();
            $this->reply(200, $result);
        }

        $name = isset($body['name']) ? (string)$body['name'] : '';
        $args = isset($body['arguments']) && is_array($body['arguments']) ? $body['arguments'] : array();
        if ($name === '') {
            fclose($sock);
            $this->error(400, 'Missing tool name');
        }

        $this->rpcSend($sock, 2, 'tools/call', array('name' => $name, 'arguments' => $args));
        $resp = $this->rpcReadUntilId($sock, 2, $this->callTimeoutMs);
        fclose($sock);
        $this->log("tools/call response", $resp);

        if ($resp === null || isset($resp['error'])) {
            $this->error(502, 'Upstream tools/call error', array('upstream_error' => isset($resp['error']) ? $resp['error'] : 'timeout'));
        }

        $result = isset($resp['result']) ? $resp['result'] : null;
        if (is_array($result) && isset($result['content'])) {
            $this->reply(200, $result);
        }
        $text = is_string($result) ? $result : json_encode($result);
        $this->reply(200, array('content' => array(array('type' => 'text', 'text' => (string)$text))));
    }

    // -------------------- Helpers (HTTP) -------------------------

    private function log($msg, $context = null)
    {
        if ($this->logFile === null) return;

        $time = date('Y-m-d H:i:s');
        $line = "[$time] $msg";
        if ($context !== null) {
            $line .= ' ' . json_encode($context, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
        }
        $line .= "\n";
        @file_put_contents($this->logFile, $line, FILE_APPEND);
    }

    private function requestMethod()
    {
        return isset($_SERVER['REQUEST_METHOD']) ? strtoupper($_SERVER['REQUEST_METHOD']) : 'GET';
    }

    private function requestPath()
    {
        if(isset($_REQUEST["path"])) {
            //for mod_rewraite usage
            return $_REQUEST["path"];
        }
        //calculate the difference between REQUEST_URI and SCRIPT_NAME
        $uri  = isset($_SERVER['REQUEST_URI']) ? $_SERVER['REQUEST_URI'] : '/';
        $scriptName = isset($_SERVER['SCRIPT_NAME']) ? $_SERVER['SCRIPT_NAME'] : '/';
        $path = substr($uri, strlen($scriptName));
        return $path ?: '/';
    }

    private function headers()
    {
        if (function_exists('getallheaders')) {
            $h = getallheaders();
            if (is_array($h)) return $h;
        }
        // Fallback
        $h = array();
        foreach ($_SERVER as $k => $v) {
            if (substr($k, 0, 5) === 'HTTP_') {
                $name = str_replace(' ', '-', ucwords(strtolower(str_replace('_', ' ', substr($k, 5)))));
                $h[$name] = $v;
            }
        }
        return $h;
    }

    private function headerValue(array $headers, $name)
    {
        foreach ($headers as $k => $v) {
            if (strcasecmp($k, $name) === 0) return $v;
        }
        return null;
    }

    private function parseBearer(array $headers)
    {
        $h = $this->headerValue($headers, 'Authorization');
        if (!$h) return null;
        if (stripos($h, 'Bearer ') === 0) return substr($h, 7);
        return null;
    }

    private function reply($status, array $payload)
    {
        http_response_code((int)$status);
        echo json_encode($payload, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
        exit;
    }

    private function error($status, $message, array $extra = array())
    {
        $err = array('error' => array('code' => (int)$status, 'message' => (string)$message) + $extra);
        $this->reply((int)$status, $err);
    }

    // -------------------- Helpers (TCP/JSON-RPC) -----------------

    private function tcpConnect($host, $port, $timeout)
    {
        $errno = 0; $errstr = '';
        $sock = @fsockopen($host, $port, $errno, $errstr, $timeout);
        if (!$sock) {
            error_log("HttpProxyController: upstream connect failed: {$host}:{$port} - {$errstr} ({$errno})");
            return null;
        }
        return $sock;
    }

    private function rpcSend($sock, $id, $method, $params)
    {
        $msg  = array('jsonrpc' => '2.0', 'id' => $id, 'method' => $method, 'params' => $params);
        $line = json_encode($msg, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE) . "\n";
        $ok = @fwrite($sock, $line);
        if ($ok === false) {
            error_log("HttpProxyController: rpcSend write failed");
        }
    }

    private function rpcReadUntilId($sock, $targetId, $deadlineMs)
    {
        $start = microtime(true);
        while (true) {
            $line = @fgets($sock);
            if ($line !== false) {
                $trim = trim($line);
                if ($trim !== '') {
                    $obj = json_decode($trim, true);
                    if (is_array($obj) && isset($obj['id']) && $obj['id'] === $targetId) {
                        return $obj;
                    }
                }
            }
            $elapsed = (microtime(true) - $start) * 1000.0;
            if ($elapsed > $deadlineMs) {
                error_log("HttpProxyController: timeout waiting for id {$targetId}");
                return null;
            }
            usleep(50000); // 50ms
        }
    }
}
