<?php
namespace boru\borumcp;

use boru\borumcp\Interfaces\AuthenticatorInterface;
use boru\borumcp\Authenticators\SharedSecretAuthenticator;
use boru\borumcp\Core\JsonRpc;
use boru\borumcp\Core\JwtSigner;
use boru\borumcp\Core\McpDispatcher;
use boru\borumcp\Core\McpSession;
use React\EventLoop\Loop;
use React\Socket\ConnectionInterface;
use React\Socket\SocketServer;
use RuntimeException;

class Server {
    private $bindIp = '127.0.0.1';
    private $bindPort = 9000;

    /** @var ToolRegistry */
    private $tools;

    /** @var AuthenticatorInterface|null */
    private $authenticator = null;

        /** @var JwtSigner|null */
    private $jwtSigner = null;

    /**
     * @param string               $bindIp
     * @param int                  $bindPort
     * @param ToolRegistry|null    $tools
     * @param AuthenticatorInterface|null   $authenticator Already-initialized authenticator (or null for none)
     * @param JwtSigner|null       $jwtSigner     Already-initialized signer (or null if you won't mint tokens)
     */
    public function __construct($bindIp = '127.0.0.1', $bindPort = 9000, $tools = null, AuthenticatorInterface $authenticator = null, JwtSigner $jwtSigner = null)
    {
        $this->bindIp   = $bindIp;
        $this->bindPort = (int)$bindPort;

        if ($tools instanceof ToolRegistry) {
            $this->tools = $tools;
        } else {
            $this->tools = MCP::toolRegistry();
        }

        // Auth is fully external / injected
        $this->authenticator = $authenticator; // may be null
        $this->jwtSigner     = $jwtSigner;     // may be null
        if($this->authenticator !== null && $this->jwtSigner === null) {
            if(method_exists($this->authenticator, 'getSigner')) {
                // If the authenticator can mint tokens, get a default signer for buildToolObjectJWT()
                $this->jwtSigner = $this->authenticator->getSigner();
            }
        }
    }

    /**
     * Start the TCP server and run the React event loop.
     */
    public function start()
    {
        $loop  = Loop::get();
        $tools = $this->tools();

        $server = new SocketServer($this->bindIp() . ':' . $this->bindPort(), array(), $loop);

        $server->on('connection', function (ConnectionInterface $conn) use ($tools) {
            $peer = $conn->getRemoteAddress();
            MCP::log("TCP client connected: " . $peer);

            // Dispatcher writes JSON-RPC lines via this function
            $sendFn = function ($line) use ($conn) {
                $conn->write($line);
            };

            $dispatcher = new McpDispatcher($tools, $sendFn);

            // Session needs a way to send early errors (e.g., 401) before dispatcher handles anything
            $writeFn = function ($obj) use ($conn) {
                $line = JsonRpc::encodeLine($obj);
                $conn->write($line);
            };
            $closeFn = function () use ($conn) {
                $conn->end();
            };

            // Session enforces "initialize with authToken" if an authenticator is provided
            $session = new McpSession($dispatcher, $this->authenticator, $closeFn);
            if (method_exists($session, 'setWriteFn')) {
                $session->setWriteFn($writeFn);
            }

            $conn->on('data', function ($data) use ($session) {
                $session->onData($data);
            });

            $conn->on('close', function () use ($peer, $session) {
                $session->flushRemainder();
                MCP::log("TCP client closed: " . $peer);
            });

            $conn->on('error', function ($e) use ($peer) {
                MCP::elog("TCP error {$peer}: " . $e->getMessage());
            });
        });

        MCP::log("TCP MCP server listening on " . $this->bindIp() . ":" . $this->bindPort());
        $loop->run();
    }

    /**
     * Stop the server loop shortly after this call.
     */
    public function stop()
    {
        Loop::addTimer(0.1, function () {
            MCP::log("Stopping server...");
            Loop::stop();
        });
    }

    /**
     * Build a single MCP tool object for the Responses API (inline host/port).
     * If a JwtSigner was injected, it will mint a short-lived JWT automatically.
     *
     * Example output:
     * {
     *   "type": "mcp",
     *   "transport": "tcp",
     *   "host": "127.0.0.1",
     *   "port": 9000,
     *   "tool_auth": { "strategy": "bearer", "token": "..." }
     * }
     *
     * @param array    $claimsExtra Extra JWT claims (merged into standard claims)
     * @param int|null $ttl         Token TTL override in seconds
     * @return array
     */
    public function buildToolObjectJWT(array $claimsExtra = array(), $ttl = null)
    {
        $tool = array(
            'type'      => 'mcp',
            'transport' => 'tcp',
            'host'      => $this->bindIp(),
            'port'      => $this->bindPort()
        );

        if ($this->jwtSigner instanceof JwtSigner) {
            $defaults = array(
                'sub'      => 'responses-api',
                'roles'    => array('mcp'),
                // Nice scoping practice: bind the token to this endpoint
                'endpoint' => $this->bindIp() . ':' . $this->bindPort()
            );
            foreach ($claimsExtra as $k => $v) {
                $defaults[$k] = $v;
            }
            $token = $this->mintToken($defaults, $ttl);
            $tool['tool_auth'] = array('strategy' => 'bearer', 'token' => $token);
        } else {
            $tool['tool_auth'] = array('strategy' => 'none');
        }

        return $tool;
    }

    /**
     * Convenience helper: build a Responses API MCP "tool" object for TCP using a freshly minted token.
     * 
     * Example Output:
     * {
     *   "type": "mcp",
     *   "server_url": "https://your-host.example.com/mcp",
     *   "server_label": "php-mcp",
     *   "allowed_tools": ["echo_text","time_now"],
     *   "require_approval": "never",
     *   "headers": {
     *     "internal-port": "9000",
     *     "Authorization": "Bearer <SHORT_LIVED_JWT>"
     *   }
     * }
     * @param array $claimsExtra
     * @param mixed $ttl
     * @return array
     * @throws RuntimeException
     */
    public function buildResponsesToolObjectJWT(array $claimsExtra = array(), $ttl = null)
    {
        if (!($this->jwtSigner instanceof JwtSigner)) {
            throw new RuntimeException("No JwtSigner configured on Server; cannot mint JWT.");
        }

        $defaults = array(
            'sub'      => 'responses-api',
            'roles'    => array('mcp'),
            // Nice scoping practice: bind the token to this endpoint
            'endpoint' => $this->bindIp() . ':' . $this->bindPort()
        );
        foreach ($claimsExtra as $k => $v) {
            $defaults[$k] = $v;
        }
        $token = $this->mintToken($defaults, $ttl);

        return array(
            'type'           => 'mcp',
            'server_url'     => "{URL-TO-PROXY-MCP}",
            'server_label'   => 'php-mcp',
            //'allowed_tools'  => array_keys($this->tools()->listTools()),
            'require_approval' => 'never',
            'headers'        => array(
                'internal-port' => (string)$this->bindPort(),
                'Authorization' => 'Bearer ' . $token
            )
        );
    }

    /**
     * Mint a short-lived JWT using the injected JwtSigner.
     *
     * @param array    $claimsExtra Extra claims to merge into the token (e.g. ['sub'=>'client1','roles'=>['mcp']])
     * @param int|null $ttl         TTL override in seconds (null -> signer default)
     * @return string               JWT string
     * @throws \RuntimeException    If no JwtSigner is configured
     */
    public function mintToken(array $claimsExtra = array(), $ttl = null)
    {
        if (!($this->jwtSigner instanceof JwtSigner)) {
            throw new \RuntimeException("No JwtSigner configured on Server; cannot mint JWT.");
        }
        return $this->jwtSigner->mint($claimsExtra, $ttl);
    }

    public function bindIp($bindIp=null) {
        if($bindIp!==null) { $this->bindIp = $bindIp; }
        return $this->bindIp;
    }
    public function bindPort($bindPort=null) {
        if($bindPort!==null) { $this->bindPort = $bindPort; }
        return $this->bindPort;
    }
    public function tools($tools=null) {
        if($tools instanceof ToolRegistry) { $this->tools = $tools; }
        elseif($tools === false) { $this->tools = new ToolRegistry(); }
        return $this->tools;
    }
    public function authenticator($authenticator=null) {
        if($authenticator instanceof AuthenticatorInterface || $authenticator===null) {
            $this->authenticator = $authenticator;
        }
        return $this->authenticator;
    }
}