<?php
namespace boru\borumcp\Core;

final class JWT
{
    public static function b64uEncode($data) {
        return rtrim(strtr(base64_encode($data), '+/', '-_'), '=');
    }
    public static function b64uDecode($b64u) {
        $b64 = strtr($b64u, '-_', '+/');
        $pad = strlen($b64) % 4;
        if ($pad) $b64 .= str_repeat('=', 4 - $pad);
        return base64_decode($b64);
    }

    /**
     * @param array $header e.g. ['alg'=>'HS256','typ'=>'JWT','kid'=>'...']
     * @param array $claims e.g. ['sub'=>'svc','iss'=>'boru','aud'=>'mcp', 'exp'=>..., 'iat'=>..., ...]
     * @param string $key   HS256: shared secret; RS256: PEM private key
     * @return string JWT
     */
    public static function encode(array $header, array $claims, $key) {
        $header  = isset($header['typ']) ? $header : $header + array('typ'=>'JWT');
        $h = self::b64uEncode(json_encode($header));
        $p = self::b64uEncode(json_encode($claims));
        $sig = self::sign("$h.$p", $header['alg'], $key);
        return "$h.$p.".self::b64uEncode($sig);
    }

    /**
     * @param string $jwt
     * @param string $alg 'HS256'|'RS256'
     * @param string $key HS256: shared secret; RS256: PEM public key
     * @return array [bool $ok, array|null $claims, string|null $err, array|null $hdr]
     */
    public static function decodeAndVerify($jwt, $alg, $key) {
        $parts = explode('.', $jwt);
        if (count($parts) !== 3) return array(false, null, 'malformed', null);
        list($hB64, $pB64, $sB64) = $parts;
        $hdrJson = self::b64uDecode($hB64);
        $pldJson = self::b64uDecode($pB64);
        if ($hdrJson === false || $pldJson === false) return array(false, null, 'bad b64', null);
        $hdr = json_decode($hdrJson, true);
        $pld = json_decode($pldJson, true);
        if (!is_array($hdr) || !is_array($pld)) return array(false, null, 'bad json', null);
        if (!isset($hdr['alg']) || strtoupper($hdr['alg']) !== strtoupper($alg)) return array(false, null, 'alg mismatch', $hdr);

        $sig = self::b64uDecode($sB64);
        $ok = self::verify("$hB64.$pB64", $sig, $alg, $key);
        if (!$ok) return array(false, null, 'bad signature', $hdr);

        return array(true, $pld, null, $hdr);
    }

    private static function sign($data, $alg, $key) {
        $alg = strtoupper($alg);
        if ($alg === 'HS256') {
            return hash_hmac('sha256', $data, $key, true);
        }
        if ($alg === 'RS256') {
            $sig = '';
            $ok = openssl_sign($data, $sig, $key, OPENSSL_ALGO_SHA256);
            if (!$ok) { throw new \RuntimeException('openssl_sign failed'); }
            return $sig;
        }
        throw new \InvalidArgumentException('Unsupported alg: '.$alg);
    }

    private static function verify($data, $sig, $alg, $key) {
        $alg = strtoupper($alg);
        if ($alg === 'HS256') {
            $calc = hash_hmac('sha256', $data, $key, true);
            return function_exists('hash_equals') ? hash_equals($calc, $sig) : $calc === $sig;
        }
        if ($alg === 'RS256') {
            return openssl_verify($data, $sig, $key, OPENSSL_ALGO_SHA256) === 1;
        }
        return false;
    }
}
