<?php
namespace boru\borumcp\Core;

use boru\borumcp\Authenticators\JWTAuthenticator;

/**
 * JwtMinter
 *
 * Mints short-lived JWTs for calling an MCP server.
 * - Defaults to RS256
 * - Pulls defaults (alg/iss/aud) from the provided JWTAuthenticator when available
 * - No external dependencies; uses Core\JWT
 *
 * Usage:
 *   $minter = new JwtMinter($authenticator, $privatePem, [
 *       'kid' => 'key-2025-10',
 *       'ttl' => 300,                // default lifetime in seconds
 *       'iss' => 'boru-mcp',         // override inferred issuer
 *       'aud' => 'mcp',              // override inferred audience
 *       'alg' => 'RS256'             // override inferred algorithm
 *   ]);
 *   $jwt = $minter->mint(['sub' => 'responses-api', 'roles' => ['mcp']], 180);
 */
final class JwtMinter
{
    /** @var JWTAuthenticator */
    private $authenticator;

    /**
     * Signing key:
     *  - RS256: private key PEM string
     *  - HS256: shared secret string
     * @var string
     */
    private $signKey;

    /** @var string 'RS256'|'HS256' */
    private $alg = 'RS256';

    /** @var string|null */
    private $kid;

    /** @var string|null */
    private $iss;

    /** @var string|string[]|null */
    private $aud;

    /** @var int default token lifetime (seconds) */
    private $ttl = 300;

    /**
     * @param JWTAuthenticator $authenticator  Used to infer alg/iss/aud (if getters exist)
     * @param string           $signKey        RS256 private key PEM or HS256 secret
     * @param array            $options        ['alg','kid','ttl','iss','aud']
     */
    public function __construct(JWTAuthenticator $authenticator, $signKey, array $options = array())
    {
        $this->authenticator = $authenticator;
        $this->signKey       = (string)$signKey;

        // Try to infer from authenticator if it exposes getters (keep BC with earlier version)
        $alg = isset($options['alg']) ? strtoupper($options['alg']) : null;
        if ($alg === null && method_exists($authenticator, 'getAlg')) {
            $alg = strtoupper($authenticator->getAlg());
        }
        $this->alg = ($alg === 'HS256' || $alg === 'RS256') ? $alg : 'RS256';

        $iss = array_key_exists('iss', $options) ? $options['iss'] : null;
        if ($iss === null && method_exists($authenticator, 'getRequiredIss')) {
            $iss = $authenticator->getRequiredIss();
        }
        $this->iss = $iss;

        $aud = array_key_exists('aud', $options) ? $options['aud'] : null;
        if ($aud === null && method_exists($authenticator, 'getRequiredAud')) {
            $aud = $authenticator->getRequiredAud();
        }
        $this->aud = $aud;

        if (isset($options['kid']) && $options['kid'] !== '') {
            $this->kid = (string)$options['kid'];
        }

        if (isset($options['ttl']) && is_int($options['ttl']) && $options['ttl'] > 0) {
            $this->ttl = (int)$options['ttl'];
        }
    }

    /**
     * Mint a JWT.
     *
     * Standard claims are prefilled (iss, aud, iat, nbf, exp) when available.
     * You can override or add any claims via $claimsExtra.
     *
     * @param array    $claimsExtra Additional claims to merge (e.g. ['sub'=>'client1','roles'=>['mcp']])
     * @param int|null $ttlOverride TTL in seconds (null => use default TTL)
     * @return string  JWT string
     */
    public function mint(array $claimsExtra = array(), $ttlOverride = null)
    {
        $now = time();
        $exp = $now + (is_int($ttlOverride) && $ttlOverride > 0 ? $ttlOverride : $this->ttl);

        $claims = array(
            // Only set iss/aud if configured/inferred; they can be null
            'iss' => $this->iss,
            'aud' => $this->aud,
            'iat' => $now,
            'nbf' => $now,
            'exp' => $exp
        );

        // Merge extra claims (allow overrides)
        foreach ($claimsExtra as $k => $v) {
            $claims[$k] = $v;
        }

        // Build header
        $header = array('alg' => $this->alg, 'typ' => 'JWT');
        if ($this->kid !== null && $this->kid !== '') {
            $header['kid'] = $this->kid;
        }

        return JWT::encode($header, $claims, $this->signKey);
    }

    /**
     * Convenience helper: build a Responses API MCP "tool" object for TCP using a freshly minted token.
     *
     * @param string $host
     * @param int    $port
     * @param array  $claimsExtra extra claims for the token
     * @param int    $ttlOverride ttl override
     * @return array MCP tool object (drop into 'tools' array)
     */
    public function buildTcpToolObject($host, $port, array $claimsExtra = array(), $ttlOverride = null)
    {
        $token = $this->mint($claimsExtra, $ttlOverride);
        return array(
            'type'      => 'mcp',
            'transport' => 'tcp',
            'host'      => (string)$host,
            'port'      => (int)$port,
            'tool_auth' => array(
                'strategy' => 'bearer',
                'token'    => $token
            )
        );
    }

    // ---- Optional getters if you want to inspect config at runtime ----

    /** @return string 'RS256'|'HS256' */
    public function getAlg() { return $this->alg; }

    /** @return string|null */
    public function getIss() { return $this->iss; }

    /** @return string|string[]|null */
    public function getAud() { return $this->aud; }

    /** @return int */
    public function getDefaultTtl() { return $this->ttl; }

    /** @return string|null */
    public function getKid() { return $this->kid; }
}
