<?php
namespace boru\borumcp\Proxy;

/**
 * InitSessionCache
 * -----------------
 * Tiny TTL cache for avoiding double-initialize in MCP proxy.
 *
 * Priority:
 *   1) APCu (if enabled)
 *   2) tmp/ file-based cache (if enabled via config)
 *   3) In-memory static array (fallback)
 */
class InitSessionCache
{
    /** @var int seconds */
    private $ttl;

    /** @var bool use tmp file cache */
    private $useTmpFiles;

    /** @var string tmp file dir */
    private $tmpDir;

    /** @var array static fallback memory */
    private static $mem = array();

    public function __construct($ttlSeconds = 8, $useTmpFiles = false, $tmpDir = '/tmp')
    {
        $this->ttl = (int)$ttlSeconds;
        if ($this->ttl <= 0) $this->ttl = 8;

        $this->useTmpFiles = (bool)$useTmpFiles;
        $this->tmpDir = rtrim($tmpDir, DIRECTORY_SEPARATOR);
    }

    public function makeKey($upstreamHost, $port, $bearer, $extras = array())
    {
        $b = $bearer !== null ? $bearer : '';
        $h = md5($b);
        $parts = array(
            'host:' . (string)$upstreamHost,
            'port:' . (string)$port,
            'bearer:' . $h,
        );
        if (is_array($extras)) {
            foreach ($extras as $k => $v) {
                $parts[] = (string)$k . ':' . (string)$v;
            }
        }
        return implode('|', $parts);
    }

    public function markInitialized($key)
    {
        $exp = microtime(true) + $this->ttl;

        // 2) tmp file cache
        if ($this->useTmpFiles) {
            $file = $this->tmpFilePath($key);
            @file_put_contents($file, (string)$exp);
            return;
        }

        // 3) memory fallback
        self::$mem[$key] = array('expires' => $exp);
        $this->cleanupMemory();
    }

    public function wasRecentlyInitialized($key)
    {
        $now = microtime(true);

        // 2) tmp file cache
        if ($this->useTmpFiles) {
            $file = $this->tmpFilePath($key);
            if (is_file($file)) {
                $exp = (float)@file_get_contents($file);
                if ($exp > $now) {
                    return true;
                }
                @unlink($file);
            }
            return false;
        }

        // 3) memory fallback
        if (isset(self::$mem[$key]) && isset(self::$mem[$key]['expires'])) {
            return self::$mem[$key]['expires'] > $now;
        }
        return false;
    }

    private function tmpFilePath($key)
    {
        $fname = 'mcp_init_' . md5($key) . '.cache';
        return $this->tmpDir . DIRECTORY_SEPARATOR . $fname;
    }

    private function cleanupMemory()
    {
        if (count(self::$mem) > 512) {
            $now = microtime(true);
            foreach (self::$mem as $k => $row) {
                if (!isset($row['expires']) || $row['expires'] < $now) {
                    unset(self::$mem[$k]);
                }
            }
        }
    }

    private function hasApcu()
    {
        return function_exists('apcu_store') && ini_get('apc.enabled');
    }
}