<?php
namespace boru\borumcp\Proxy;

use boru\borumcp\Proxy\HttpResponder;

class McpJsonRpcBridgeController
{
    /** @var \boru\borumcp\ProxyConfig */
    private $cfg;

    public function __construct($cfg) { $this->cfg = $cfg; }

    public function handle($ctx)
    {
        // 1) SSE handling (GET /, Accept: text/event-stream)
        if ($this->isSseRequest($ctx)) {
            $this->openSse($ctx);
            return; // never returns
        }

        // 2) Validate that we are POST /
        if (!$this->isJsonRpcPost($ctx)) {
            $this->log('error', 'Invalid request to JSON-RPC bridge', array(
                'method' => $ctx->method,
                'path'   => $ctx->path,
                'remote' => isset($ctx->remoteAddr) ? $ctx->remoteAddr : 'unknown'
            ));
            HttpResponder::error(404, 'Not Found');
        }

        // 3) Parse JSON-RPC request (object tree)
        $req = $this->parseJsonRpcOrFail($ctx);

        // 4) Normalize id / method / params
        $rpcId   = $this->extractId($req);
        $rpcMeth = (string)$req->method;
        $params  = $this->normalizeParams($req);

        // 4a) JSON-RPC notification (no id): do not forward, just acknowledge
        if ($rpcId === null) {
            // example: notifications/initialized
            $this->log('cli->app', 'Notification received', array(
                'method' => $rpcMeth,
                'remote' => isset($ctx->remoteAddr) ? $ctx->remoteAddr : 'unknown'
            ));
            // 200 OK with a tiny body (safe across clients)
            HttpResponder::json(200, array('ok' => true));
        }
        // 5) Resolve upstream port & connect
        $port = $this->resolvePortOrFail($ctx);
        $cli  = $this->connectUpstreamOrFail($port);

        // 6) Handle initialize directly (pass-through; add protocol/auth if missing)
        if ($rpcMeth === 'initialize') {
            $params = $this->ensureInitFields($params, $ctx);
            $resp   = $this->sendAndAwait($cli, $rpcId, 'initialize', $params, $this->cfg->initTimeoutMs);
            $cli->close();

            if ($resp === null) {
                $this->log('error', 'Upstream timeout (initialize)', array('id' => $rpcId));
                HttpResponder::error(502, 'Upstream timeout (initialize)');
            }
            $resp = $this->sanitizeInitResponse($resp);

            $this->log('app->cli', 'To client', array('id'=>$rpcId,'method'=>$rpcMeth,'response'=>$resp));
            HttpResponder::json(200, $resp);
        }

        // 7) Stateless model: ALWAYS initialize this new TCP connection first
        $this->initializeOrFail($cli, $ctx);

        // 8) Forward the real client RPC and return
        $this->log('app->pro', 'Forwarding RPC', array('method'=>$rpcMeth,'id'=>$rpcId));
        $deadline = $this->deadlineFor($rpcMeth);
        $resp     = $this->sendAndAwait($cli, $rpcId, $rpcMeth, $params, $deadline);
        $cli->close();

        if ($resp === null) {
            $this->log('error', 'Upstream timeout ('.$rpcMeth.')', array('id'=>$rpcId));
            HttpResponder::error(502, 'Upstream timeout ('.$rpcMeth.')');
        }

        $this->log('pro->app', 'Upstream response received', array('id'=>$rpcId,'method'=>$rpcMeth));
        $this->log('app->cli', 'To client', array('id'=>$rpcId,'method'=>$rpcMeth,'response'=>$resp));
        HttpResponder::json(200, $resp);
    }

    // -------------------- Small helpers --------------------

    private function isSseRequest($ctx)
    {
        return $ctx->method === 'GET' && $ctx->path === '/' && $ctx->wantsEventStream();
    }

    private function openSse($ctx)
    {
        $remote = isset($ctx->remoteAddr) ? $ctx->remoteAddr : 'unknown';
        (new SseStreamer($this->cfg->logger))->keepAliveLoop($remote);
    }

    private function isJsonRpcPost($ctx)
    {
        return ($ctx->method === 'POST' && $ctx->path === '/');
    }

    private function parseJsonRpcOrFail($ctx)
    {
        $req = json_decode($ctx->bodyRaw ? $ctx->bodyRaw : 'null');
        if (!is_object($req) || !isset($req->jsonrpc) || !isset($req->method)) {
            $this->log('error', 'Invalid JSON-RPC request at root', array(
                'body'   => $ctx->bodyRaw,
                'remote' => isset($ctx->remoteAddr) ? $ctx->remoteAddr : 'unknown'
            ));
            HttpResponder::error(400, 'Invalid JSON-RPC request at root');
        }
        return $req;
    }

    private function extractId($req)
    {
        return property_exists($req, 'id') ? $req->id : null;
    }

    private function normalizeParams($req)
    {
        // Keep params as object; default to {}
        $params = isset($req->params) ? $req->params : (object)[];

        // Ensure capabilities is an object if present
        if (isset($params->capabilities) && is_array($params->capabilities)) {
            if (empty($params->capabilities)) {
                $params->capabilities = (object)[];
            } else {
                $params->capabilities = (object)$params->capabilities;
            }
        }
        return $params;
    }

    // Normalize initialize result so capabilities.* are objects, not arrays.
    private function sanitizeInitResponse($resp)
    {
        // $resp is an assoc array from TcpJsonRpcClient::readUntilId (json_decode(true) or false?)
        // In your code it’s an array; if it’s stdClass, adjust access accordingly.

        if (!is_array($resp)) return $resp;
        if (!isset($resp['result']) || !is_array($resp['result'])) return $resp;

        $res = &$resp['result'];
        if (!isset($res['capabilities']) || !is_array($res['capabilities'])) return $resp;

        $caps = &$res['capabilities'];

        foreach (array('tools','resources','prompts') as $k) {
            if (isset($caps[$k])) {
                // If it’s an array, coerce empty [] to {} and non-empty to object-like cast.
                if (is_array($caps[$k])) {
                    if (empty($caps[$k])) {
                        $caps[$k] = (object)array(); // {}
                    } else {
                        // Convert associative array to object; numeric arrays become keyed props (harmless here)
                        $caps[$k] = (object)$caps[$k];
                    }
                }
            } else {
                // Absent -> provide empty object to signal support
                $caps[$k] = (object)array();
            }
        }

        return $resp;
    }

    private function resolvePortOrFail($ctx)
    {
        $hdrName = $this->cfg->internalPortHeader;
        $portHeader = $ctx->headerValue($hdrName);
        if ($portHeader === null || $portHeader === '' || !ctype_digit($portHeader)) {
            HttpResponder::error(400, 'Missing or invalid header: '.$hdrName);
        }
        $port = (int)$portHeader;
        if ($port < $this->cfg->minPort || $port > $this->cfg->maxPort) {
            HttpResponder::error(403, 'Port not allowed', array(
                'allowed_range' => $this->cfg->minPort.'-'.$this->cfg->maxPort
            ));
        }
        return $port;
    }

    private function connectUpstreamOrFail($port)
    {
        $cli = new TcpJsonRpcClient($this->cfg->logger);
        if (!$cli->connect($this->cfg->upstreamHost, $port, $this->cfg->connectTimeout, $this->cfg->readTimeout)) {
            $this->log('error', 'Upstream TCP connect failed', array('port'=>$port));
            HttpResponder::error(502, 'Upstream TCP connect failed', array('port'=>$port));
        }
        return $cli;
    }

    private function ensureInitFields($params, $ctx)
    {
        if (!isset($params->protocolVersion)) {
            $params->protocolVersion = $this->cfg->protocolVersion;
        }
        $bearer = $ctx->bearerOrNull();
        if ($bearer !== null && !isset($params->authToken)) {
            $params->authToken = $bearer;
        }
        return $params;
    }

    private function initializeOrFail($cli, $ctx)
    {
        $bearer = $ctx->bearerOrNull();
        // Always init this connection (stateless HTTP -> new TCP socket)
        $resp = $cli->initialize($this->cfg->protocolVersion, $bearer, $this->cfg->initTimeoutMs);
        // TcpJsonRpcClient::initialize() already HttpResponder::error(...) on failure.
        // We can still log a success marker if desired:
        $this->log('pro->app', 'Upstream initialized (stateless)', array('id'=>1));
        return $resp;
    }

    private function deadlineFor($rpcMeth)
    {
        return (strpos($rpcMeth, 'tools/') === 0) ? $this->cfg->callTimeoutMs : $this->cfg->listTimeoutMs;
    }

    private function sendAndAwait($cli, $id, $method, $params, $deadlineMs)
    {
        $cli->send($id, $method, $params);
        return $cli->readUntilId($id, $deadlineMs);
    }

    private function log($tag, $msg, $ctx)
    {
        if ($this->cfg->logger) {
            // your FileLogger prepends timestamp/caller; we pass a compact tag in $message
            $this->cfg->logger->log($tag, $msg, (is_array($ctx) ? $ctx : array()));
        }
    }
}
