<?php
namespace boru\borumcp\Proxy;

use boru\borumcp\Logger\LoggerInterface;

final class ToolsRestController
{
    private $cfg;
    public function __construct(ProxyConfig $cfg) { $this->cfg = $cfg; }

    public function handle(HttpContext $ctx)
    {
        if ($ctx->method === 'GET' && ($ctx->path === '/health' || $ctx->path === '/mcp/health')) {
            HttpResponder::json(200, ['ok'=>true,'status'=>'healthy']);
        }

        if ($ctx->method !== 'POST') HttpResponder::error(404, 'Not Found');
        if ($ctx->path !== '/tools/list' && $ctx->path !== '/tools/call') {
            HttpResponder::error(404, 'Not Found');
        }

        $port = $this->resolvePortOrFail($ctx);

        $cli = new TcpJsonRpcClient($this->cfg->logger);
        if (!$cli->connect($this->cfg->upstreamHost, $port, $this->cfg->connectTimeout, $this->cfg->readTimeout)) {
            HttpResponder::error(502, 'Upstream TCP connect failed', ['port' => $port]);
        }

        $bearer = $ctx->bearerOrNull();
        $cli->initialize($this->cfg->protocolVersion, $bearer, $this->cfg->initTimeoutMs);

        if ($ctx->path === '/tools/list') {
            $cli->send(2, 'tools/list', new \stdClass());
            $resp = $cli->readUntilId(2, $this->cfg->listTimeoutMs);
            $cli->close();

            if ($resp === null || isset($resp['error'])) {
                HttpResponder::error(502, 'Upstream tools/list error', [
                    'upstream_error' => $resp['error'] ?? 'timeout'
                ]);
            }
            $result = (isset($resp['result']) && is_array($resp['result'])) ? $resp['result'] : [];
            HttpResponder::json(200, $result);
        }

        // /tools/call
        $body = ($ctx->bodyRaw !== '') ? json_decode($ctx->bodyRaw, true) : [];
        if (!is_array($body)) HttpResponder::error(400, 'Expected JSON body');

        $name = isset($body['name']) ? (string)$body['name'] : '';
        $args = (isset($body['arguments']) && is_array($body['arguments'])) ? $body['arguments'] : [];
        if ($name === '') {
            $cli->close();
            HttpResponder::error(400, 'Missing tool name');
        }

        $cli->send(2, 'tools/call', ['name'=>$name,'arguments'=>$args]);
        $resp = $cli->readUntilId(2, $this->cfg->callTimeoutMs);
        $cli->close();

        if ($resp === null || isset($resp['error'])) {
            HttpResponder::error(502, 'Upstream tools/call error', [
                'upstream_error' => $resp['error'] ?? 'timeout'
            ]);
        }

        $result = $resp['result'] ?? null;
        if (is_array($result) && isset($result['content'])) {
            HttpResponder::json(200, $result);
        }
        $text = is_string($result) ? $result : json_encode($result);
        HttpResponder::json(200, ['content' => [['type'=>'text','text'=>(string)$text]]]);
    }

    private function resolvePortOrFail(HttpContext $ctx)
    {
        $portHeader = $ctx->headerValue($this->cfg->internalPortHeader);
        if ($portHeader === null || $portHeader === '' || !ctype_digit($portHeader)) {
            HttpResponder::error(400, 'Missing or invalid header: '.$this->cfg->internalPortHeader);
        }
        $port = (int)$portHeader;
        if ($port < $this->cfg->minPort || $port > $this->cfg->maxPort) {
            HttpResponder::error(403, 'Port not allowed', ['allowed_range' => $this->cfg->minPort.'-'.$this->cfg->maxPort]);
        }
        return $port;
    }
}
