<?php
namespace boru\borumcp\Proxy;

use boru\borumcp\Logger\LoggerInterface;

final class McpJsonRpcBridgeController
{
    private $cfg;

    public function __construct(ProxyConfig $cfg) { $this->cfg = $cfg; }

    public function handle(HttpContext $ctx)
    {
        // GET / with SSE accept
        if ($ctx->method === 'GET' && $ctx->path === '/' && $ctx->wantsEventStream()) {
            (new SseStreamer($this->cfg->logger))->keepAliveLoop($ctx->remoteAddr ?? 'unknown');
        }

        // POST / with JSON-RPC payload
        if (!($ctx->method === 'POST' && $ctx->path === '/')) {
            if($this->cfg->logger) {
                $this->cfg->logger->log("Invalid request to JSON-RPC bridge", [
                    'method' => $ctx->method,
                    'path'   => $ctx->path,
                    'remote' => $ctx->remoteAddr ?? 'unknown'
                ]);
            }
            HttpResponder::error(404, 'Not Found');
        }

        $req = json_decode($ctx->bodyRaw ?: 'null'); // stdClass (object) tree
        if (!is_object($req) || !isset($req->jsonrpc) || !isset($req->method)) {
            if($this->cfg->logger) {
                $this->cfg->logger->log("Invalid JSON-RPC request at root", [
                    'body' => $ctx->bodyRaw,
                    'remote' => $ctx->remoteAddr ?? 'unknown'
                ]);
            }
            HttpResponder::error(400, 'Invalid JSON-RPC request at root');
        }

        $rpcId = property_exists($req, 'id') ? $req->id : 1; // 1 as fallback only if truly absent

        $rpcMeth = (string)$req->method;
        // Params: keep as object; default to {}
        $params = isset($req->params) ? $req->params : (object)[];

        // Some JSON encoders send capabilities: [] but spec expects object; coerce if needed
        if (isset($params->capabilities) && is_array($params->capabilities)) {
            // Convert [] to {} without changing populated maps
            if (empty($params->capabilities)) {
                $params->capabilities = (object)[];
            } else {
                // If somehow an array with numeric keys arrived, turn into object to be safe
                $params->capabilities = (object)$params->capabilities;
            }
        }


        $port = $this->resolvePortOrFail($ctx);

        $cli = new TcpJsonRpcClient($this->cfg->logger);
        if (!$cli->connect($this->cfg->upstreamHost, $port, $this->cfg->connectTimeout, $this->cfg->readTimeout)) {
            if($this->cfg->logger) {
                $this->cfg->logger->log("Upstream TCP connect failed", ['port' => $port]);
            }
            HttpResponder::error(502, 'Upstream TCP connect failed', ['port' => $port]);
        }

        if($this->cfg->logger) {
            $this->cfg->logger->log("Incoming {$ctx->method} {$ctx->path}", [
                'body' => $ctx->bodyRaw,
                'remote_addr' => $ctx->remoteAddr
            ]);
        }
        $bearer = $ctx->bearerOrNull();

        if ($rpcMeth === 'initialize') {
            if (!isset($params->protocolVersion)) {
                $params->protocolVersion = $this->cfg->protocolVersion;
            }
            $bearer = $ctx->bearerOrNull();
            if ($bearer !== null && !isset($params->authToken)) {
                $params->authToken = $bearer;
            }

            // send exactly what the client intended (now with corrected shapes)
            $cli->send($rpcId, 'initialize', $params);
            $resp = $cli->readUntilId($rpcId, $this->cfg->initTimeoutMs);
            $cli->close();
            if ($resp === null) {
                if($this->cfg->logger) {
                    $this->cfg->logger->log("Upstream timeout (initialize)", ['id'=>$rpcId]);
                }
                HttpResponder::error(502, 'Upstream timeout (initialize)');
            }
            if($this->cfg->logger) {
                $this->cfg->logger->log("Upstream initialized", ['id'=>$rpcId,'response'=>$resp]);
            }
            HttpResponder::json(200, $resp);
        }
        // Stateless: init each request
        $bearer = $ctx->bearerOrNull();
        $cli->initialize($this->cfg->protocolVersion, $bearer, $this->cfg->initTimeoutMs);

        // Send the real method with the client's id (0, "a", etc. are OK)
        if ($this->cfg->logger) {
            $this->cfg->logger->log('Forwarding RPC', array('method' => $rpcMeth, 'id' => $rpcId));
        }
        $cli->send($rpcId, $rpcMeth, $params);
        $deadline = (strpos($rpcMeth, 'tools/') === 0) ? $this->cfg->callTimeoutMs : $this->cfg->listTimeoutMs;
        $resp = $cli->readUntilId($rpcId, $deadline);
        $cli->close();

        if ($resp === null) HttpResponder::error(502, 'Upstream timeout ('.$rpcMeth.')');
        if($this->cfg->logger) {
            $this->cfg->logger->log("Upstream response received", ['id'=>$rpcId,'method'=>$rpcMeth]);
            $this->cfg->logger->log("Upstream response", $resp);
        }
        HttpResponder::json(200, $resp);
    }

    private function resolvePortOrFail(HttpContext $ctx)
    {
        $portHeader = $ctx->headerValue($this->cfg->internalPortHeader);
        if ($portHeader === null || $portHeader === '' || !ctype_digit($portHeader)) {
            HttpResponder::error(400, 'Missing or invalid header: '.$this->cfg->internalPortHeader);
        }
        $port = (int)$portHeader;
        if ($port < $this->cfg->minPort || $port > $this->cfg->maxPort) {
            HttpResponder::error(403, 'Port not allowed', ['allowed_range' => $this->cfg->minPort.'-'.$this->cfg->maxPort]);
        }
        return $port;
    }
}
