<?php
namespace boru\borumcp\Proxy;

use boru\borumcp\Logger\LoggerInterface;

final class TcpJsonRpcClient
{
    /** @var resource|null */
    private $sock = null;
    private $readTimeout;

    /** @var LoggerInterface|null */
    private $logger;

    public function __construct(?LoggerInterface $logger = null) { $this->logger = $logger; }

    public function connect(string $host, int $port, float $connectTimeout, float $readTimeout)
    {
        $errno = 0; $errstr = '';
        $this->sock = @fsockopen($host, $port, $errno, $errstr, $connectTimeout);
        if (!$this->sock) {
            if ($this->logger) $this->logger->log("Upstream connect failed", ['host'=>$host,'port'=>$port,'errno'=>$errno,'err'=>$errstr]);
            return false;
        }
        stream_set_blocking($this->sock, true);
        stream_set_timeout($this->sock, $readTimeout);
        $this->readTimeout = $readTimeout;
        if ($this->logger) $this->logger->log("Upstream connected", ['host'=>$host,'port'=>$port]);
        return true;
    }

    public function close()
    {
        if ($this->sock) {
            @fclose($this->sock);
            $this->sock = null;
            if ($this->logger) $this->logger->log("Upstream connection closed");
        }
    }

    private function normalizeId($id) {
        // Valid ids are string | int | null. Don't coerce 0.
        if (is_int($id) || is_string($id) || $id === null) return $id;
        // Fallback for exotic types: stringify to keep deterministic matching
        return (string)$id;
    }

    public function send($id, string $method, $params)
    {
        if (!$this->sock) return;
        $msg  = ['jsonrpc' => '2.0', 'id' => $this->normalizeId($id), 'method' => $method, 'params' => $params];
        $line = json_encode($msg, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE) . "\n";
        @fwrite($this->sock, $line);
        if ($this->logger) $this->logger->log("Sending upstream", ['line'=>$line]);
    }

    /** @return array<string,mixed>|null */
    public function readUntilId($targetId, int $deadlineMs)
    {
        if (!$this->sock) return null;
        $start = microtime(true);
        while (true) {
            $line = @fgets($this->sock);
            if ($line !== false) {
                $trim = trim($line);
                if ($trim !== '') {
                    $obj = json_decode($trim, true);
                    if (is_array($obj) && isset($obj['id']) && $obj['id'] === $targetId) {
                        if ($this->logger) {
                            $this->logger->log('Upstream RPC response', array('id' => $targetId, 'obj' => $obj));
                        }
                        return $obj;
                    }
                }
            }
            $elapsed = (microtime(true) - $start) * 1000.0;
            if ($elapsed > $deadlineMs) {
                if ($this->logger) $this->logger->log("Timeout waiting for id", ['id'=>$targetId,'deadlineMs'=>$deadlineMs]);
                return null;
            }
            usleep(50_000);
        }
    }

    /** Performs initialize and returns upstream response or throws via HttpResponder::error */
    public function initialize(string $protocolVersion, ?string $bearer, int $initTimeoutMs)
    {
        $params = ['protocolVersion' => $protocolVersion];
        if ($bearer !== null) $params['authToken'] = $bearer;

        $this->send(1, 'initialize', $params);
        $resp = $this->readUntilId(1, $initTimeoutMs);
        if ($this->logger) $this->logger->log("Init response", $resp ? $resp : ['timeout'=>true]);

        if ($resp === null || isset($resp['error'])) {
            $this->close();
            if($this->logger) {
                $this->logger->log("Upstream initialize failed", ['error'=>$resp['error'] ?? 'timeout']);
            }
            HttpResponder::error(401, 'Upstream initialize failed', [
                'upstream_error' => $resp['error'] ?? 'timeout'
            ]);
        }
        if($this->logger) $this->logger->log("Upstream initialized", ['response'=>$resp]);
        return $resp;
    }
}
