<?php
namespace boru\vtigertools\loader;

use boru\dhdb\dhDB;
use boru\dhutils\dhGlobal;
use boru\dhutils\filesys\Directory;
use boru\dhutils\filesys\File;

class Loader {
    /** @var Directory */
    private $vtigerDirectory;
    /** @var VersionInterface */
    private $vtigerVersion;
    /** @var string */
    private $vtigerVersionString;

    private $includes = [];

    public function __construct($sourceDirectory=null) {
        if(($this->vtigerDirectory = $this->findVtigerRootDirectory($sourceDirectory)) === false) {
            throw new \Exception("Unable to find vtiger within the path provided.");
        }
        $this->vtigerVersionString = $this->detectVtigerVersion();
        $versionClass = "";
        if(!empty($this->vtigerVersionString)) {
            $t = explode(".",$this->vtigerVersionString);
            $versionClass = "\\boru\\vtigertools\\loader\\versions\\Vtiger".$t[0].$t[1]."0";
            if(class_exists($versionClass)) {
                $this->vtigerVersion = new $versionClass($this->vtigerDirectory);
            }
        }
        if(is_null($this->vtigerVersion)) {
            throw new \Exception("Unable to load VtigerVersion file $versionClass");
        }
        $this->includes = $this->vtigerVersion->getIncludes();
        $this->initDb();

        if($this->vtigerVersion->needsHTMLPurifierFix()) {
            $this->HTMLPurifierFix($this->vtigerVersion->getHtmlPurifierFile());
        }
    }

    private function initDb() {
        $db = dhDB::fromVtigerConfig($this->vtigerDirectory->path("config.inc.php"));
        dhGlobal::set("dhDB",$db);
    }

    private function detectVtigerVersion() {
        $versionFile = new File(["path"=>$this->vtigerDirectory->path("vtigerversion.php")]);
        $contents = $versionFile->content(["lineFilter"=>function($line) {
            if(substr($line,0,strlen('$vtiger_current_version')) == '$vtiger_current_version') {
                return $line;
            } else {
                return false;
            }
        }]);
        if(!empty($contents)) {
            $arr = explode("=",$contents,2);
            if(isset($arr[1])) {
                $val = dhGlobal::rtrimString(";",$arr[1]);
                $val = dhGlobal::btrimString(" ",$val);
                $val = dhGlobal::btrimString("'",$val);
                $val = dhGlobal::ltrimString(":",$val);
                return $val;
            } else {
                throw new \Exception("Unable to identify vtiger version");
            }
        }
    }


    /**
     * Find the VTiger directory starting with $directory (__DIR__ if not provided)
     * @param null|string
     * @return string|false 
     */
    private function findVtigerRootDirectory($directory=null) {
        if(is_null($directory) || empty($directory)) {
            $directory = __DIR__;
        }
        $dirParts = explode(DIRECTORY_SEPARATOR,$directory);
                
        if(($dir = $this->checkIsVtigerDirectory($directory)) !== false) {
            //config.inc.php exists in this directory.. weird but ok..
            return $dir;
        }

        $continue=true;
        while($continue) {
            $dropoff = array_pop($dirParts);
            if(empty($dirParts)) {
                $continue=false;
                break;
            }
            $dirPath = implode(DIRECTORY_SEPARATOR,$dirParts);
            if(($dir = $this->checkIsVtigerDirectory($dirPath)) !== false) {
                return $dir;
            }
        }
        return false;
    }
    private function checkIsVtigerDirectory($directoryPathString) {
        $filesRequiredForVtiger = [
            "config.inc.php",
            "index.php",
            "vtigerversion.php",
            "config.php",
            "tabdata.php",
        ];

        if(($dir = dhGlobal::dirIfExists($directoryPathString)) !== false) {
            foreach($filesRequiredForVtiger as $fileName) {
                if($dir->file($fileName) === false) {
                    return false;
                }
            }
            return $dir;
        }
        return false;
    }

    public function HTMLPurifierFix($fileName) {
        $file = dhGlobal::fileIfExists($this->vtigerDirectory->path($fileName));
        if($file !== false) {
            $hasChanged = false;
            $contents = $file->content(["lineFilter"=>function($line) use (&$hasChanged) {
                $matches = [];
                if(preg_match("/^(\s*)include_once \('(include|libraries)\/htmlpurifier\/library\/HTMLPurifier.auto.php'\);/",$line,$matches) === 1) {
                    if(count($matches)>1) {
                        $line = str_replace("HTMLPurifier.auto.php","HTMLPurifier/Bootstrap.php",$line);
                        $line.="\n";
                        $line.=$matches[1]."spl_autoload_register(array('HTMLPurifier_Bootstrap', 'autoload'));";
                        $hasChanged=true;
                    }
                }
                return $line;
            }]);
            if($hasChanged) {
                $file->write($contents);
            }
        }
        
    }

    /**
	 * Get the value of includes
     * @return  array
     */
    public function getIncludes() {
        return $this->includes;
    }

    /**
	 * Get the value of vtigerDirectory
     * @return  Directory
     */
    public function getVtigerDirectory() {
        return $this->vtigerDirectory;
    }

    /**
	 * Get the value of vtigerVersion
     * @return  mixed
     */
    public function getVtigerVersion() {
        return $this->vtigerVersion;
    }
}