<?php
namespace boru\cli;

use boru\cli\commands\HelpCommand;
use boru\cli\models\Args;
use boru\cli\models\ClassCommand;
use boru\cli\models\Command;
use boru\cli\models\Commands;
use boru\cli\models\Params;
use boru\cli\params\Help;
use boru\cli\params\Positional;
use boru\cli\traits\CreateParams;
use boru\dhutils\dhGlobal;
use boru\dhutils\tools\DebugTrace;

class CLI {
    /**
     * This trait is used to create params
     * ->positional($syntaxString,$opts=[])
     * ->flag($syntaxString,$opts=[])
     * ->option($syntaxString,$opts=[])
     */
    use CreateParams;

    /** @var string */
    private $type = "cli";
    /** @var string */
    private $name;
    /** @var string */
    private $description;
    /** @var Args */
    private $args;
    /** @var Params */
    private $params;
    /** @var array */
    private $results;
    /** @var Commands */
    private $commands;

    private $helpCommand;

    private $callback;

    public function __construct($syntaxOrArray="", $params=null, $callback=null) {
        $this->helpCommand(HelpCommand::class);
        $this->commands(false);
        $this->commands()->callback(function($command) {
            if($command === false) {
                $this->print(null,["error"=>"No command specified or invalid command"]);
            } else {
                $this->print(null,["error"=>"Command completed successfully, but callback was either not set or returned true to continue"]);
            }
        });
        if(is_array($syntaxOrArray)) {
            $this->setData($syntaxOrArray);
        } elseif(is_string($syntaxOrArray)) {
            $this->parseSyntaxString($syntaxOrArray);
        }
        if($params !== null && $params !== false) {
            $this->params($params);
        }
        if($callback !== null && $callback !== false) {
            $this->callback($callback);
        }
    }

    public function type($type=null) {
        if($type === null) {
            return $this->type;
        }
        $this->type = $type;
        return $this;
    }
    public function name($name=null) {
        if($name === null) {
            return $this->name;
        }
        $this->name = $name;
        return $this;
    }
    public function description($description=null) {
        if($description === null) {
            return $this->description;
        }
        $this->description = $description;
        return $this;
    }

    public function args($args=null) {
        if($args === null) {
            return $this->args;
        }
        if($args === false) {
            $this->args = new Args();
            return $this;
        }
        $this->args = new Args($args);
        return $this;
    }

    /**
     * @param Params|array|null $params
     * @return Params|CLI
     */
    public function params($params=null) {
        if($params === null) {
            return $this->params;
        }
        if($params === false) {
            $this->params = new Params();
            $this->params->add(Help::create());
            $this->params->cli($this);
            return $this;
        }
        array_unshift($params,Help::create());
        $this->params = new Params($params);
        $this->params->cli($this);
        return $this;
    }

    /**
     * @param Commands|array|null $commands
     * @return Commands|CLI
     */
    public function commands($commands=null) {
        if($commands === null) {
            return $this->commands;
        }
        if($commands === false) {
            $this->commands = new Commands();
            $this->commands->cli($this);
            $this->commands->add(new ClassCommand($this->helpCommand));
            return $this;
        }
        $this->commands = new Commands($commands);
        $this->commands->cli($this);
        return $this;
    }

    public function results($results=null) {
        if($results === null) {
            return $this->results;
        }
        $this->results = $results;
        return $this;
    }

    public function callback($callback=null) {
        if($callback === null) {
            return $this->callback;
        }
        $this->callback = $callback;
        $this->commands()->callback($callback);
        return $this;
    }

    public function helpCommand($className=null) {
        if($className === null) {
            return $this->helpCommand;
        }
        if(is_object($className)) {
            $className = get_class($className);
        }
        $this->helpCommand = $className;
        return $this;
    }

    public function get($name=null,$default=null) {
        if($name === null) {
            return $this->results;
        }
        return dhGlobal::dotGet($this->results,$name,$default);
    }
    public function set($name,$value) {
        if(strpos($name,".") !== false) {
            dhGlobal::dotAssign($this->results,$name,$value);
        }
        else {
            $this->results[$name] = $value;
        }
    }
    public function exists($column=null) {
        if(is_null($column)) {
            return !empty($this->modelData);
        }
        return !is_null(dhGlobal::dotGet($this->results,$column,null));
    }
    public function remove($column) {
        return dhGlobal::dotDelete($this->results,$column);
    }

    public function parseSyntaxString($syntaxString) {
        $parts = explode("|",$syntaxString);
        if(count($parts) < 2) {
            throw new \Exception("Invalid syntax string, must have at least 2 parts separated by |");
        }
        $opts["type"] = "cli";
        $opts["syntax"] = $syntaxString;
        $opts["name"] = $parts[0];
        $opts["description"] = $parts[1];
        $this->setData($opts);
    }

    public function add($command) {
        $this->commands()->add($command);
        return $this;
    }

    public function setData($array) {
        if(isset($array["type"])) {
            $this->type($array["type"]);
        }
        if(isset($array["name"])) {
            $this->name($array["name"]);
        }
        if(isset($array["description"])) {
            $this->description($array["description"]);
        }
        if(isset($array["args"])) {
            $this->args($array["args"]);
        }
        if(isset($array["params"])) {
            $this->params($array["params"]);
        }
        if(isset($array["results"])) {
            $this->results($array["results"]);
        }
    }
    
    /**
     * Parse the arguments
     * @param array $args
     * @return true|array
     */
    public function parse($args=null) {
        if(php_sapi_name() !== "cli") {
            $msg = static::class."->parse() must be run from the command line";
            $divider = str_repeat("-",strlen($msg)+2);
            throw new \Exception("\n\n".$divider."\n ".static::class."::parse() must be run from the command line\n".$divider."\n\n");
        }
        if($args !== null) {
            $this->args($args);
        } elseif($this->args() === null) {
            $this->args([]);
        }
        if(($result = $this->parseCommonParams()) !== true) {
            return $result;
        }
        if(($result = $this->parseCommands()) !== true) {
            return $result;
        }
    }

    private function parseCommonParams() {
        if($this->params() === null) {
            return true;
        }
        $this->params()->cli($this);
        if(($result = $this->params()->parse($this->args())) === true) {
            $this->results($this->params()->results());
        } else {
            return $result;
        }
        return true;
    }

    private function parseCommands() {
        //unconsumed args
        $args = $this->args()->unused();
        return $this->commands()->parse($args,$this->results());
    }

    public function syntax($commandString=null) {
        $script = $this->args()->script();
        $syntax = "";
        if($script === null) {
            $syntax = $this->name();
        } else {
            $script = basename($script);
        }
        $syntax = $script;
        if(substr($script,-4) === ".php") {
            $syntax = "php ".$script;
        }
        if($this->params() !== null) {
            $paramSyntax = $this->params()->syntax();
            if($paramSyntax !== "") {
                $syntax .= " ".$paramSyntax;
            }
        }
        if($commandString !== null) {
            $syntax .= " ".$commandString;
        } else {
            $syntax .= " <command>";
        }
        
        return $syntax;
    }
    public function banner($printer) {
        $printer->line("{INDENT}".$this->name());
        $printer->line("{INDENT}{INDENT}".$this->description());
        $printer->line("");
        return $this;
    }
    public function error($error,$printer) {
        $printer->addLine("");
        $printer->line("Error: ".$error);
        $printer->addLine("");
        return $this;
    }
    public function print($printer=null,$options=[]) {
        $defaults = [
            "banner"=>false,
            "error"=>null,
            "syntax"=>true,
            "commands"=>true,
            "params"=>true,
            "printer"=>null
        ];
        $options = array_merge($defaults,$options);
        $output = false;
        if($printer === null) {
            $printer = new Printer();
            $output = true;
        }
        $printer->header(1);
        if($options["banner"]) {
            $this->banner($printer);
        }
        if($options["error"] !== null) {
            $this->error($options["error"],$printer);
        }
        if($options["syntax"]) {
            $printer->line("Syntax: ".$this->syntax());
        }
        if($options["printer"]) {
            $printer->fromPrinter($options["printer"]);
        }
        if($options["commands"]) {
            $printer->separator();
            $this->commands()->print($printer);
            if($options["params"] && $this->params() !== null) {
                $printer->separator();
                $this->params()->print($printer);
            }
        }
        $printer->footer(1);
        if($output) {
            $printer->print();
        }
    }

    public static function relPath($from, $to, $ps = DIRECTORY_SEPARATOR) {
        $arFrom = explode($ps, rtrim($from, $ps));
        $arTo = explode($ps, rtrim($to, $ps));
        while(count($arFrom) && count($arTo) && ($arFrom[0] == $arTo[0])) {
            array_shift($arFrom);
            array_shift($arTo);
        }
        if(empty($arFrom) && count($arTo) === 1) {
            return ".".$ps.$arTo[0];
        }
        return str_pad("", count($arFrom) * 3, '..'.$ps).implode($ps, $arTo);
    }
}