<?php
namespace boru\cli;

use boru\cli\models\Commands;
use boru\cli\models\Args;
use boru\cli\models\Command;

/**
 * This class will read all methods prefixed with "cmd_" and create a command object for each one. There will need to be 3 total methods for each command:
 * - cmd_commandName() - The method name will be the command name, and the method will be the callback for the command
 * - cmd_commandName_desc() - The method name will be the command name with _desc appended, and the method will return the description of the command
 * - cmd_commandName_params() - The method name will be the command name with _params appended, and the method will return the parameters for the command
 * @property Commands $commands
 * @property Args $args
 */
abstract class CLIInterface {
    /** @var callable|false */
    private static $CLI_HELP_CALLBACK = false;

    public static function parse($args=null) {
        $def = static::CLI_def();
        if($def instanceof CLI) {
            $cli = $def;
        } else {
            if(isset($def["errorCallback"])) {
                static::$CLI_HELP_CALLBACK = $def["errorCallback"];
                unset($def["errorCallback"]);
            } elseif(isset($def["error"])) {
                static::$CLI_HELP_CALLBACK = $def["error"];
                unset($def["error"]);
            } elseif(isset($def["error_callback"])) {
                static::$CLI_HELP_CALLBACK = $def["error_callback"];
                unset($def["error_callback"]);
            }
            $cli = new CLI($def);
        }
        $commandsArray = static::CLI_generateCommands();
        foreach($commandsArray as $command) {
            $cli->add($command);
        }
        static::$cli = $cli;
        $cli->parse($args);
        
    }

    /** @var Command */
    private $command;
    public function __construct($command) {
        $this->command = $command;
    }
    public function get($key,$default=null) {
        return $this->command->get($key,$default);
    }
    public function set($key,$value) {
        $this->command->set($key,$value);
        return $this;
    }

    /**
     * Returns the args object.
     * @return Args
     */
    public function args() {
        return $this->command->args();
    }
    /**
     * Returns the params object.
     * @return Params
     */
    public function params() {
        return $this->command->params();
    }
    /**
     * Returns the results array
     * @return array
     */
    public function results() {
        return $this->command->results();
    }

    /**
     * Define the CLI. This will be the main method that will configure the CLI tool. Must return an array with 'name', 'description', and optionally 'params'. Alternatively you can return a CLI object.
     * @return array|CLI
     */
    abstract public static function CLI_def();

    protected static $cli;

    public static function getCli() {
        if(static::$cli === null) {
            static::$cli = new CLI();
        }
        return static::$cli;
    }

    public static function CLI_successCallback($command) {
        if(method_exists(static::class,"cmd_".$command->name())) {
            $object = new static($command);
            $commandName = "cmd_".$command->name();
            echo "Running command $commandName\n";
            $result = call_user_func_array([$object,$commandName],[$command]);
            echo "Command $commandName finished\n";
        }
    }
    public static function CLI_errorCallback($command) {
        if(static::$CLI_HELP_CALLBACK) {
            return call_user_func_array(static::$CLI_HELP_CALLBACK,[$command]);
        }
    }

    private static function CLI_generateCommands() {
        $methods = get_class_methods(static::class);
        $commandArray = [];
        foreach($methods as $method) {
            if(strpos($method,"def_") === 0) {
                $commandName = substr($method,4);
                $def = static::$method();
                $commandArray[$commandName]['description'] = $def['description'];
                $commandArray[$commandName]['params'] = $def['params'];
            }
            if(strpos($method,"cmd_") === 0) {
                $commandName = substr($method,4);
                if(substr($method,-5) === "_def") {
                    $commandName = substr($method,4,-5);
                    $def = static::$method();
                    $commandArray[$commandName]['description'] = $def['description'];
                    $commandArray[$commandName]['params'] = $def['params'];
                } elseif(substr($method,-5) === "_desc") {
                    $commandName = substr($method,4,-5);
                    $commandArray[$commandName]['description'] = static::$method();
                } elseif(substr($method,-7) === "_params") {
                    $commandName = substr($method,4,-7);
                    $commandArray[$commandName]['params'] = static::$method();
                } else {
                    $commandArray[$commandName]['callback'] = [static::class,$method];
                }
            }
        }
        return static::CLI_makeCommands($commandArray);
    }
    private static function CLI_validateCommandsArray($commandArray) {
        foreach($commandArray as $commandName => $commandData) {
            if(!isset($commandData['description'])) {
                return static::CLI_exception_def($commandName);
                throw new \Exception("Command $commandName is missing a description.. Please add a method called cmd_".$commandName."_desc() that returns the description");
            }
            if(!isset($commandData['callback'])) {
                return static::CLI_exception_def($commandName);
                throw new \Exception("Command $commandName is missing a callback.. Please add a method called cmd_".$commandName."() that is the callback for the command");
            }
            if(!isset($commandData['params'])) {
                return static::CLI_exception_def($commandName);
                throw new \Exception("Command $commandName is missing parameters.. Please add a method called cmd_".$commandName."_params() that returns the parameters for the command");
            }
        }
    }
    private static function CLI_makeCommands($commandArray) {
        static::CLI_validateCommandsArray($commandArray);
        $commands = [];
        foreach($commandArray as $commandName => $commandData) {
            $command = new Command(
                [
                    "name"=>$commandName,
                    "description"=>$commandData['description']
                ],
                $commandData['params'],
                [static::class,"CLI_successCallback"],
                [static::class,"CLI_errorCallback"]
            );
            $commands[] = $command;
        }
        return $commands;
    }
    private static function CLI_exception_def($commandName) {
        throw new \Exception("Error in ".static::class."::CLI_generateCommands()
-----------------------------------------------------------------------------------------------            
    Command $commandName is missing a definition..
    Please add a method called def_".$commandName."() that returns an array with a description and params
    the description should be a string and the params should be an array of parameters
    [
        'description' => 'Description of the command',
        'params' => [
            new Param('name','description')
        ]
    ]
-----------------------------------------------------------------------------------------------\n\n");
        return null;
    }
}