<?php
namespace boru\cli\models;

use boru\cli\traits\CreateParams;
use boru\dhutils\dhGlobal;
use boru\cli\CLI;
use boru\cli\Printer;

class Command {


    /**
     * This trait is used to create params
     * ->positional($syntaxString,$opts=[])
     * ->flag($syntaxString,$opts=[])
     * ->option($syntaxString,$opts=[])
     */
    use CreateParams;

    /** @var string */
    private $type = "command";
    /** @var string */
    private $name;
    /** @var string */
    private $description;
    /** @var Args */
    private $args;
    /** @var Params */
    private $params;
    /** @var array */
    private $results = [];
    /** @var CLI */
    private $cli;
    /** @var Commands */
    private $commands;
    /** @var Command */
    private $parent;
    

    private $callback;
    private $errorCallback;

    public function __construct($syntaxOrArray="", $params=null, $callback=null, $errorCallback=null) {
        $this->results = [];
        if(is_array($syntaxOrArray)) {
            $this->setData($syntaxOrArray);
        } elseif(is_string($syntaxOrArray)) {
            $this->parseSyntaxString($syntaxOrArray);
        }
        if($params !== null && $params !== false) {
            $this->params($params);
        }
        if($callback !== null && $callback !== false) {
            $this->callback($callback);
        }
        if($errorCallback !== null && $errorCallback !== false) {
            $this->errorCallback($errorCallback);
        }
    }

    public function type($type=null) {
        if($type === null) {
            return $this->type;
        }
        $this->type = $type;
        return $this;
    }
    public function name($name=null) {
        if($name === null) {
            return $this->name;
        }
        $this->name = $name;
        return $this;
    }
    public function description($description=null) {
        if($description === null) {
            return $this->description;
        }
        $this->description = $description;
        return $this;
    }

    public function args($args=null) {
        if($args === null) {
            return $this->args;
        }
        if($args === false) {
            $this->args = new Args();
            return $this;
        }
        $this->args = new Args($args);
        return $this;
    }

    public function params($params=null) {
        if($params === null) {
            return $this->params;
        }
        if($params === false) {
            $this->params = new Params();
            $this->params->command($this);
            return $this;
        }
        $this->params = new Params($params);
        $this->params->command($this);
        return $this;
    }

    public function results($results=null,$merge=true) {
        if($results === null) {
            return $this->results;
        }
        if($results === false) {
            $this->results = [];
            return $this;
        }
        if($merge) {
            $this->results = array_merge($this->results,$results);
            return $this;
        }
        $this->results = $results;
        return $this;
    }

    public function callback($callback=null) {
        if($callback === null) {
            return $this->callback;
        }
        $this->callback = $callback;
        return $this;
    }

    public function errorCallback($errorCallback=null) {
        if($errorCallback === null) {
            return $this->errorCallback;
        }
        $this->errorCallback = $errorCallback;
        return $this;
    }

    public function cli($cli=null) {
        if($cli === null) {
            return $this->cli;
        }
        $this->cli = $cli;
        return $this;
    }

    public function parent($parent=null) {
        if($parent === null) {
            return $this->parent;
        }
        $this->parent = $parent;
        return $this;
    }

    public function commands($commands=null) {
        if($commands === null) {
            return $this->commands;
        }
        if($commands === false) {
            $this->commands = new Commands();
            return $this;
        }
        $this->commands = new Commands($commands);
        return $this;
    }

    public function get($name=null,$default=null) {
        if($name === null) {
            return $this->results;
        }
        return dhGlobal::dotGet($this->results,$name,$default);
    }
    public function set($name,$value) {
        if(strpos($name,".") !== false) {
            dhGlobal::dotAssign($this->results,$name,$value);
        }
        else {
            $this->results[$name] = $value;
        }
    }
    public function exists($column=null) {
        if(is_null($column)) {
            return !empty($this->modelData);
        }
        return !is_null(dhGlobal::dotGet($this->results,$column,null));
    }
    public function remove($column) {
        return dhGlobal::dotDelete($this->results,$column);
    }

    public function parseSyntaxString($syntaxString) {
        $parts = explode("|",$syntaxString);
        if(count($parts) < 2) {
            throw new \Exception("Invalid syntax string, must have at least 2 parts separated by |");
        }
        $opts["type"] = "cli";
        $opts["syntax"] = $syntaxString;
        $opts["name"] = $parts[0];
        $opts["description"] = $parts[1];
        $this->setData($opts);
    }

    public function setData($array) {
        if(isset($array["type"])) {
            $this->type($array["type"]);
        }
        if(isset($array["name"])) {
            $this->name($array["name"]);
        }
        if(isset($array["description"])) {
            $this->description($array["description"]);
        }
        if(isset($array["args"])) {
            $this->args($array["args"]);
        }
        if(isset($array["params"])) {
            $this->params($array["params"]);
        }
        if(isset($array["results"])) {
            $this->results($array["results"]);
        }
    }
    
    /**
     * Parse the arguments
     * @param array $args
     * @return true|array
     */
    public function parse($args=null,$previousResults=null) {
        if($args !== null) {
            $this->args($args);
        }
        if($previousResults !== null) {
            $this->results($previousResults);
        }
        $result = $this->params()->parse($this->args());
        $this->results(array_merge($this->results(),$this->params()->results()));
        if($result === false) {
            return false;
        }
        if($result !== true) {
            if($this->errorCallback()) {
                $callbackResult = $this->errorCallback()($result,$this);
                if($callbackResult === true) {
                    return $this;
                }
            }
            $this->printError($result);
            return false;
        }
        if($result === true && $this->callback()) {
            return $this->callback()($this);
        }
        return $result;
    }

    public function printError($errors,$printer=null) {
        $output = false;
        if($printer === null) {
            $output = true;
            $printer = new Printer();
            $printer->header(1);
        }
        $printer->addLine("Error processing command ".$this->name());
        $table = $printer->table();
        if(is_array($errors)) {
            foreach($errors as $error) {
                $table->row("",$error)->setWidth(0,1);
            }
        }
        $printer->line();
        $printer->addLine("Syntax: ".$this->syntax());
        if($output) {
            $printer->footer(0,1);
            $printer->print();
        }
    }
    public function syntax() {
        $syntax = $this->name();
        if($this->cli()) {
            $syntax = $this->cli()->syntax($this->name());
        }
        if($this->params()) {
            $syntax .= " ".$this->params()->syntax();
        }
        return $syntax;
    }
    public function print($printer) {
        $printer->addLine($this->name()." - ".$this->description());
        return true;
    }

    public static function create($syntaxOrArray="", $params=null, $callback=null) {
        return new Command($syntaxOrArray, $params, $callback);
    }
}