<?php
namespace boru\cli\models;

use boru\cli\params\Flag;
use boru\cli\params\Option;
use boru\cli\params\Positional;
use boru\cli\Printer;
use boru\cli\traits\CreateParams;

class Params extends Model {
    /**
     * This trait is used to create params
     * ->positional($syntaxString,$opts=[])
     * ->flag($syntaxString,$opts=[])
     * ->option($syntaxString,$opts=[])
     */
    use CreateParams;

    public function initData() {
        $this->iteratorElement("params");
        $this->set('params',[]);
        $this->set("type","params");
        $this->set("description","");
        $this->set("callback",null);
        $this->set("placeholder","<value>");
        $this->set("paramNames",[]);
        $this->set("command",null);
        $this->set("cli",null);
        $this->set("args",null);
    }

    public function __construct($params=[],$callback=null) {
        $this->initData();
        if($params instanceof Params) {
            $this->setData($params->toArray());
        } elseif($params !== null && $params !== false) {
            $this->params($params);
        }
        if($callback !== null && $callback !== false) {
            $this->callback($callback);
        }
    }

    public function callback($callback=null) {
        if($callback === null) {
            return $this->get('callback');
        }
        if($callback === false) {
            $this->set('callback',null);
            return $this;
        }
        $this->set('callback',$callback);
        return $this;
    }

    public function results() {
        $results = [];
        $params = $this->params();
        foreach($params as $param) {
            //if($param->value() !== null) {
                $results[$param->get("name")] = $param->value();
            //}
        }
        return $results;
    }

    public function required($type=null) {
        $params = $this->params();
        return array_filter($params,function($param) use ($type) {
            return $param->required() && ($type === null || $param->type() === $type);
        });
    }
    public function optional($type=null) {
        $params = $this->params();
        return array_filter($params,function($param) use ($type) {
            return !$param->required() && ($type === null || $param->type() === $type);
        });
    }
    public function flags($required=null) {
        $params = $this->params();
        return array_filter($params,function($param) use ($required) {
            return $param->type() === "flag" && ($required === null || $param->required() === $required);
        });
    }
    public function positional($required=null) {
        $params = $this->params();
        return array_filter($params,function($param) use ($required) {
            return $param->type() === "positional" && ($required === null || $param->required() === $required);
        });
    }
    public function options($required=null) {
        $params = $this->params();
        return array_filter($params,function($param) use ($required) {
            return $param->type() !== "flag" && $param->type() !== "positional" && ($required === null || $param->required() === $required);
        });
    }

    /**
     * 
     * @param string $name 
     * @return Param[] 
     */
    public function params($params=null) {
        if($params === null) {
            return $this->get('params',[]);
        }
        if($params === false) {
            $this->set('params',[]);
            return $this;
        }
        $this->set('params',$params);
        return $this;
    }

    public function description($description=null) {
        if($description === null) {
            return $this->get('description');
        }
        if($description === false) {
            $this->set('description','');
            return $this;
        }
        $this->set('description',$description);
        return $this;
    }

    public function command($command=null) {
        if($command === null) {
            return $this->get('command');
        }
        $this->set('command',$command);
        if($command !== null) {
            if($command->cli() !== null) {
                $this->cli($command->cli());
            }
        }
        return $this;
    }
    public function cli($cli=null) {
        if($cli === null) {
            return $this->get('cli');
        }
        $this->set('cli',$cli);
        return $this;
    }

    public function paramNames($paramNames=null) {
        if($paramNames === null) {
            return $this->get('paramNames');
        }
        if($paramNames === false) {
            $this->set('paramNames',[]);
            return $this;
        }
        $this->set('paramNames',$paramNames);
        return $this;
    }

    public function add($param) {
        if(is_array($param)) {
            foreach($param as $p) {
                $this->add($p);
            }
            return $this;
        }
        if($param instanceof Params) {
            foreach($param->params() as $p) {
                $this->add($p);
            }
            return $this;
        }
        $params = $this->params();
        $params[] = $param;
        $param->params($this);
        $this->params($params);
        if($param->get("name") !== null) {
            $paramNames = $this->paramNames();
            $paramNames[$param->get("name")] = $param;
            $this->paramNames($paramNames);
        }
    }
    
    
    /**
     * 
     * @param string $name 
     * @return Param[]|null 
     */
    public function getType($type) {
        return array_filter($this->params(),function($param) use ($type) {
            return $param->get("type") === $type;
        });
    }

    /**
     * 
     * @param Args|array $args
     * @return true|array 
     */
    public function parse($args=null) {
        if($args === null) {
            $args = new Args();
        } elseif(is_array($args)) {
            $args = new Args($args);
        } else {
            $args = new Args($args);
        }
        while(($arg = $args->next())) {
            $this->parseArg($arg,$args->next(false));
        }
        $errors = [];
        foreach($this->params() as $param) {
            $param->validate();
            if($param->error() !== false) {
                $errors[] = $param->error();
            }
        }
        if(!empty($errors)) {
            return $errors;
        }
        foreach($this->params() as $param) {
            if($param->callback() !== null && $param->consumed()) {
                $callback = $param->callback();
                if($callback($param) === false) {
                    return false;
                }
            }
        }
        if($this->callback() !== null) {
            $callback = $this->callback();
            return $callback($this);
        }
        return true;
    }

    private $params = [];
    private $positionalParams = [];

    private function parseArg($arg,$nextArg=null) {
        if(empty($this->params) && empty($this->positionalParams)) {
            $allParams = $this->params();
            foreach($allParams as $param) {
                $param->params($this);
                if($param->get("type") === "positional") {
                    $this->positionalParams[] = $param;
                } else {
                    $this->params[] = $param;
                }
            }
        }

        foreach($this->params as $k=>$param) {
            if($param->value() !== null && $param->multiple() === false) {
                unset($this->params[$k]);
            } else {                
                if($param->parseArg($arg,$nextArg)) {
                    return;
                }
            }
        }
        foreach($this->positionalParams as $k=>$param) {
            if($param->parseArg($arg,$nextArg)) {
                return;
            }
        }
    }

    public function syntax() {
        $strings = [];
        $flags = [];
        foreach($this->flags() as $flag) {
            $flags[] = $flag->syntax();
        }
        if(!empty($flags)) {
            $strings[] = "[-".implode("",$flags)."]";
        }

        $options = [];
        foreach($this->options(true) as $option) {
            $options[] = $option->syntax();
        }
        if(!empty($options)) {
            $strings[] = implode(" ",$options);
        }

        $positionals = [];
        foreach($this->positional(true) as $positional) {
            $positionals[] = $positional->syntax();
        }
        if(!empty($positionals)) {
            $strings[] = implode(" ",$positionals);
        }
        if(!empty($this->options(false)) || !empty($this->positional(false))) {
            $strings[] = "[options]";
        }
        
        return implode(" ",$strings);
    }
    /**
     * 
     * @param Printer|null $printer 
     * @return void 
     */
    public function print($printer=null) {
        $output = false;
        if($printer === null) {
            $printer = new Printer();
            $output = true;
        }
        if(!empty($this->required())) {
            $printer->addLine("Required Parameters:");
            $table = $printer->table();
            foreach($this->required() as $param) {
                $short = $param->short() ? $param->shortDisplay() : "";
                $long = $param->name() ? $param->longDisplay() : "";
                $table->row("",$short,"",$long,"--",$param->description())->setWidth(0,1);
            }
        }
        if(!empty($this->optional())) {
            if(!empty($this->required())) {
                $printer->line();
            }
            $printer->addLine("Optional Parameters:");
            $table = $printer->table();
            foreach($this->optional() as $param) {
                $short = $param->short() ? $param->shortDisplay() : "";
                $long = $param->name() ? $param->longDisplay() : "";
                $table->row("",$short,"",$long,"--",$param->description())->setWidth(0,1);
            }
        }
        if($output) {
            $printer->print();
        }
    }
    
}