<?php
namespace boru\cli;

use boru\cli\models\Commands;
use boru\cli\models\Args;
use boru\cli\models\Command;
use boru\cli\models\CommandGroup;

/**
 * This class will read all methods prefixed with "cmd_" and create a command object for each one. There will need to be 3 total methods for each command:
 * - cmd_commandName() - The method name will be the command name, and the method will be the callback for the command
 * - cmd_commandName_desc() - The method name will be the command name with _desc appended, and the method will return the description of the command
 * - cmd_commandName_params() - The method name will be the command name with _params appended, and the method will return the parameters for the command
 * @property Commands $commands
 * @property Args $args
 */
abstract class CLIInterface {
    /** @var callable|false */
    private static $CLI_HELP_CALLBACK = false;
    private static $CLI_OBJECT = null;
    public static function CLI_getCli() {
        if(self::$CLI_OBJECT === null) {
            $def = static::CLI_def();
            if($def instanceof CLI) {
                static::$CLI_OBJECT = $def;
            } else {
                if(isset($def["errorCallback"])) {
                    static::$CLI_HELP_CALLBACK = $def["errorCallback"];
                    unset($def["errorCallback"]);
                } elseif(isset($def["error"])) {
                    static::$CLI_HELP_CALLBACK = $def["error"];
                    unset($def["error"]);
                } elseif(isset($def["error_callback"])) {
                    static::$CLI_HELP_CALLBACK = $def["error_callback"];
                    unset($def["error_callback"]);
                }
                self::$CLI_OBJECT = new CLI($def);
            }
        }
        return self::$CLI_OBJECT;
    }
    public static function parse($args=null) {
        $cli = static::getCli();
        $commandsArray = static::CLI_generateCommands();
        foreach($commandsArray as $command) {
            $cli->add($command);
        }
        static::$cli = $cli;
        $cli->parse($args);
        
    }

    /** @var Command */
    private $command;
    public function __construct($command) {
        $this->command = $command;
    }
    public function get($key,$default=null) {
        return $this->command->get($key,$default);
    }
    public function set($key,$value) {
        $this->command->set($key,$value);
        return $this;
    }

    /**
     * Returns the args object.
     * @return Args
     */
    public function args() {
        return $this->command->args();
    }
    /**
     * Returns the params object.
     * @return Params
     */
    public function params() {
        return $this->command->params();
    }
    /**
     * Returns the results array
     * @return array
     */
    public function results() {
        return $this->command->results();
    }

    /**
     * Define the CLI. This will be the main method that will configure the CLI tool. Must return an array with 'name', 'description', and optionally 'params'. Alternatively you can return a CLI object.
     * @return array|CLI
     */
    abstract public static function CLI_def();

    protected static $cli;
    protected static $CLI_callback_map = [];

    public static function getCli() {
        if(static::$cli === null) {
            static::$cli = static::CLI_getCli();
        }
        return static::$cli;
    }

    public static function CLI_successCallback($method,$command) {
        if(method_exists(static::class,$method)) {
            $object = new static($command);
            $result = call_user_func_array([$object,$method],[$command]);
        }
    }
    public static function CLI_errorCallback($command) {
        if(static::$CLI_HELP_CALLBACK) {
            return call_user_func_array(static::$CLI_HELP_CALLBACK,[$command]);
        }
    }

    private static function CLI_generateCommands() {
        $methods = get_class_methods(static::class);
        $commandArray = [];
        $defsArray = [];
        $callbackArray = [];
        foreach($methods as $method) {
            if(substr($method,0,4) === 'CLI_') {
                continue;
            }
            if(strpos($method,"def_") === 0) {
                $commandName = substr($method,4);
                $def = static::$method();
                $defsArray[$commandName] = $def;
            }
            if(strpos($method,"cmd_") === 0) {
                $commandName = substr($method,4);
                $callbackArray[$commandName] = $method;
            }
        }
        foreach($callbackArray as $commandName=>$callback) {
            if(strpos($commandName,"_") !== false) {
                $parts = explode("_",$commandName);
                if(count($parts)>2) {
                    throw new \Exception("Subcommand depth is too deep. Only 1 level of subcommands is allowed");
                }
                $parentCommand = $parts[0];
                $subCommand = $parts[1];
                if(isset($defsArray[$parentCommand]) && !isset($commandArray[$parentCommand])) {
                    $commandArray[$parentCommand] = $defsArray[$parentCommand];
                }
                if(isset($defsArray[$commandName])) {
                    $commandArray[$parentCommand]["commands"][$subCommand] = $defsArray[$commandName];
                }
                $commandArray[$parentCommand]["commands"][$subCommand]["callback"] = $callback;
            } else {
                if(isset($defsArray[$commandName])) {
                    $commandArray[$commandName] = $defsArray[$commandName];
                }
                $commandArray[$commandName]["callback"] = $callback;
            }
        }
        return static::CLI_makeCommands($commandArray);
    }
    private static function CLI_validateCommandsArray($commandArray) {
        foreach($commandArray as $commandName => $commandData) {
            if(!isset($commandData['description'])) {
                return static::CLI_exception_def($commandName);
            }
            if(!isset($commandData['callback']) && !isset($commandData['commands'])) {
                return static::CLI_exception_def($commandName);
            }
            if(!isset($commandData['params'])) {
                $commandArray[$commandName]['params'] = [];
            }
            if(isset($commandData['commands'])) {
                foreach($commandData['commands'] as $subCommandName => $subCommandData) {
                    if(!isset($subCommandData['description'])) {
                        return static::CLI_exception_def($commandName."_".$subCommandName);
                    }
                    if(!isset($subCommandData['callback'])) {
                        return static::CLI_exception_def($commandName."_".$subCommandName);
                    }
                    if(!isset($subCommandData['params'])) {
                        $commandArray[$commandName]['commands'][$subCommandName]['params'] = [];
                    }
                }
            }
        }
    }
    private static function CLI_makeCommands($commandArray) {
        static::CLI_validateCommandsArray($commandArray);
        $commands = [];
        foreach($commandArray as $commandName => $commandData) {
            if(isset($commandData["commands"])) {
                static::CLI_makeCommandGroup($commandName,$commandData);
            } else {
                $commands[] = static::CLI_makeCommand($commandName,$commandData);
            }
        }
        return $commands;
    }
    private static function CLI_makeCommand($commandName,$commandData) {
        if(!isset($commandData["params"]) || !is_array($commandData["params"])) {
            $commandData["params"] = [];
        }
        $command = new Command(
            [
                "name"=>$commandName,
                "description"=>$commandData['description']
            ],
            $commandData['params'],
            function($command) use($commandData) {
                static::CLI_successCallback($commandData["callback"],$command);
            },
            [static::class,"CLI_errorCallback"]
        );
        return $command;
    }
    private static function CLI_makeCommandGroup($commandName,$commandData) {
        $cli = static::getCli();
        if(!isset($commandData["params"]) || !is_array($commandData["params"])) {
            $commandData["params"] = [];
        }
        $group = $cli->commandGroup($commandName."|".$commandData['description'],
            $commandData["params"]
        );
        foreach($commandData["commands"] as $subCommandName => $subCommandData) {
            if(!isset($subCommandData["params"]) || !is_array($subCommandData["params"])) {
                $subCommandData["params"] = [];
            }
            $group->command($subCommandName."|".$subCommandData['description'],$subCommandData["params"],function($command) use($subCommandData) {
                static::CLI_successCallback($subCommandData["callback"],$command);
            });
        }
    }
    private static function CLI_exception_def($commandName) {
        throw new \Exception("Error in ".static::class."::CLI_generateCommands()
-----------------------------------------------------------------------------------------------            
    Command $commandName is missing a definition..
    Please add a method called def_".$commandName."() that returns an array with a description and params
    the description should be a string and the params should be an array of parameters
    Example:

    function def_".$commandName."() {
        return [
            'description' => 'Description of the command',
            'params' => [
                Flat::create('p|print|Print the output'),
            ]
        ];
    }
-----------------------------------------------------------------------------------------------\n\n");
        return null;
    }
}