<?php
namespace boru\cli;

use boru\cli\printer\Line;
use boru\cli\printer\Lines;
use boru\cli\printer\Table;
use boru\dot\Dot;
use boru\output\Output;

class Printer {
    private $indent = 2;
    private $longestLine = 0;

    /** @var Output */
    private $output;

    /** @var Lines */
    private $lines;
    private $options = [];

    /** @var BasePrinter */
    private $printer;

    public function __construct($lines=null) {
        $this->output = new Output();
        $this->configureOutput("stdout",true);
        $this->configureOutput("prefix",false);
        $this->configureOutput("prefixFormat","{PREFIX}");
        $this->lines($lines);
    }

    public function output($output=null) {
        if($output === null) {
            return $this->output;
        }
        $this->output = $output;
        return $this;
    }

    public function lines($lines=null) {
        if($this->lines === null) {
            $this->lines = new Lines($this);
        }
        if($lines === null) {
            return $this->lines;
        }
        $this->lines = $lines;
        return $this;
    }

    public function indent($indent=null) {
        if($indent === null) {
            return $this->indent;
        }
        $this->indent = $indent;
        return $this;
    }

    public function get($key,$default=null) {
        return Dot::get($this->options,$key,$default);
    }
    public function set($key,$value) {
        Dot::set($this->options,$key,$value);
        return $this;
    }

    public function configure($what,$value=null) {
        if(is_array($what)) {
            foreach($what as $key => $val) {
                $this->configureOutput($key,$val);
            }
        } else {
            $this->configureOutput($what,$value);
        }
    }

    public function configureOutput($what,$value) {
        if($what == "stdout") {
            $this->output->option("stdOut",$value);
        }
        if($what == "file") {
            $this->output->option("fileName",$value);
        }
        if($what == "prefix") {
            if($value) {
                $this->output->option("prefix",true);
                $this->output->option("prefixFormat",$value);
            } else {
                $this->output->option("prefix",false);
            }
        }
        if($what == "eol") {
            $this->output->option("lineSeparator",$value);
        }
    }

    public function addLine($line,$type="line") {
        $this->lines->addLine($line,$type);
    }
    public function line($text="") {
        $this->addLine(Line::line($text,$this));
    }

    /**
     * Add a line to the output from a printer.
     * @param Printer $printer
     * @return $this
     */
    public function fromPrinter($printer) {
        $this->lines()->fromLines($printer->lines());
        return $this;
    }

    /**
     * Add a table to the output and return the table.
     * @param Table|null $table The table to add. If null, a new table is created.
     * @return Table
     */
    public function table($table=null) {
        if($table === null || !($table instanceof Table)) {
            $table = new Table();
        }
        $this->lines->addLine($table,"table");
        return $table;
    }

    public function makeSeparator($character="-",$length=null) {
        if($length === null) {
            $length = $this->longestLine();
        }
        return str_repeat($character,$length);
    }
    public function makeIndent($indent=null) {
        if($indent === null) {
            $indent = $this->indent();
        }
        return str_repeat(" ",$indent);
    }

    public function header($rowsAfer=0,$rowsBefore=0) {
        for($i=0;$i<$rowsBefore;$i++) {
            $this->addLine("");
        }
        $this->addLine("{HEADER}");
        for($i=0;$i<$rowsAfer;$i++) {
            $this->addLine("");
        }
    }
    public function footer($rowsAfer=0,$rowsBefore=0) {
        for($i=0;$i<$rowsBefore;$i++) {
            $this->addLine("");
        }
        $this->addLine("{FOOTER}");
        for($i=0;$i<$rowsAfer;$i++) {
            $this->addLine("");
        }
    }
    public function separator() {
        $this->addLine("{SEPARATOR}");
    }
    public function registerLength($length) {
        if($length > $this->longestLine) {
            $this->longestLine = $length;
        }
    }
    public function longestLine($length=null) {
        if($length === null) {
            return $this->longestLine;
        }
        $this->longestLine = $length;
        return $this;
    }


    public function print($options=[]) {
        $this->lines->parse();
        foreach($this->lines->lines() as $line) {
            $this->output->line($line->__toString());
        }
    }
}