<?php
namespace boru\cli\models;

use boru\cli\CLI;
use Exception;

abstract class Param extends Model {

    /**
     * Parse the argument
     * 
     * @param Arg $arg The argument
     * @param Arg|null $nextArg The next argument
     * 
     * @return bool
     */
    abstract public function parseArg($arg,$nextArg=null);

    public function initData() {
        $this->set('name',null);
        $this->set('short',null);
        $this->set('value',null);
        $this->set("type","param");
        $this->set('consumed',false);
        $this->set("options",[]);
        $this->set("format",'/^(.)+$/');
        $this->set("description",null);
        $this->set("required",false);
        $this->set("multiple",false);
        $this->set("default",null);
        $this->set("needsValue",true);
        $this->set("callback",null);
        $this->set("error",false);
        $this->set("params",null);
        $this->set("finished",false);
    }
    public function __construct($input=null) {
        $this->initData();
        if($input instanceof Param) {
            $this->setData($input->toArray());
            return;
        }
        if(is_array($input)) {
            $this->setData($input);
            return;
        }
    }

    /**
     * @param string|null $value
     * @return string|this
     */
    public function name($value=null) {
        if($value === null) {
            return $this->get('name');
        }
        $this->set('name',$value);
        return $this;
    }

    /**
     * @param string|null $value
     * @return string|bool|this
     */
    public function short($value=null) {
        if($value === null) {
            $short = $this->get('short');
            if($short === null) {
                return false;
            }
            return $short;
        }
        $this->set('short',$value);
        return $this;
    }

    /**
     * @param string|null $value
     * @return string|this
     */
    public function value($value=null) {
        if($value === null) {
            return $this->get('value');
        }
        if($this->multiple()) {
            if(!is_array($this->value())) {
                $this->set('value',[]);
            }
            $this->set('value',array_merge($this->value(),[$value]));
            return $this;
        }
        $this->set('value',$value);
        return $this;
    }

    /**
     * @param bool|null $value
     * @return bool|this
     */
    public function consumed($value=null) {
        if($value === null) {
            return $this->get('consumed',false) ? true : false;
        }
        $this->set('consumed',$value ? true : false);
        return $this;
    }

    /**
     * Consume the parameter
     * 
     * @return bool
     */
    public function consume() {
        $this->consumed(true);
        return true;
    }

    /**
     * @param string|null $value
     * @return string|this
     */
    public function type($value=null) {
        if($value === null) {
            return $this->get('type');
        }
        $this->set('type',$value);
        return $this;
    }

    /**
     * @param array|null $value
     * @return array|this
     */
    public function options($value=null) {
        if($value === null) {
            return $this->get('options');
        }
        $this->set('options',$value);
        return $this;
    }

    /**
     * @param string|null $value
     * @return string|this
     */
    public function format($value=null) {
        if($value === null) {
            return $this->get('format');
        }
        $this->set('format',$value);
        return $this;
    }

    /**
     * @param string|null $value
     * @return string|this
     */
    public function description($value=null) {
        if($value === null) {
            return $this->get('description');
        }
        $this->set('description',$value);
        return $this;
    }

    /**
     * @param bool|null $value
     * @return bool|this
     */
    public function required($value=null) {
        if($value === null) {
            return $this->get('required',false) ? true : false;
        }
        $this->set('required',$value ? true : false);
        return $this;
    }

    /**
     * @param bool|null $value
     * @return bool|this
     */
    public function multiple($value=null) {
        if($value === null) {
            return $this->get('multiple',false) ? true : false;
        }
        $this->set('multiple',$value ? true : false);
        return $this;
    }
    /**
     * @param bool|null $value
     * @return bool|this
     */
    public function finished($value=null) {
        if($value === null) {
            return $this->get('finished',false) ? true : false;
        }
        $this->set('finished',$value ? true : false);
        return $this;
    }

    /**
     * @param string|null $value
     * @return string|this
     */
    public function default($value=null) {
        if($value === null) {
            return $this->get('default',null);
        }
        $this->set('default',$value);
        return $this;
    }

    /**
     * @param bool|null $value
     * @return bool|this
     */
    public function needsValue($value=null) {
        if($value === null) {
            return $this->get('needsValue',true) ? true : false;
        }
        $this->set('needsValue',$value ? true : false);
        return $this;
    }

    /**
     * @param callable|null $value
     * @return callable|this
     */
    public function callback($value=null) {
        if($value === null) {
            return $this->get('callback');
        }
        $this->set('callback',$value);
        return $this;
    }

    /**
     * @param string|false $value
     * @return string|this
     */
    public function error($value=null) {
        if($value === null) {
            return $this->get('error',false);
        }
        $this->set('error',$value);
        return $this;
    }

    /**
     * @param Params|null $value
     * @return Params|this
     */
    public function params($value=null) {
        if($value === null) {
            return $this->get('params');
        }
        $this->set('params',$value);
        return $this;
    }

    /**
     * Get the command
     * 
     * @return Command|null
     */
    public function command() {
        return $this->params()->command();
    }
    /**
     * Get the CLI instance
     * 
     * @return CLI|null
     */
    public function cli() {
        if($this->params() === null) {
            return null;
        }
        return $this->params()->cli();
    }

    public function validate() {
        $value = $this->value();
        if($this->required() && $value === null) {
            //throw new \Exception("'{$this->name()}' is required");
            $this->error("'{$this->name()}' is required");
            return false;
        }
        if($value === null) {
            $value = $this->default();
        }
        if($value === null) {
            $value = "";
        }
        if($this->format() !== null && !empty($value)) {
            $valArr = is_array($value) ? $value : [$value];
            foreach($valArr as $val) {
                if(!preg_match($this->format(),$val)) {
                    $this->error("'{$this->name()}' must match format '{$this->format()}'");
                    return false;
                }
            }
            /*if(!preg_match($this->format(),$value)) {
                $this->error("'{$this->name()}' must match format '{$this->format()}'");
                return false;
            }*/
        }
        if($this->options() && is_array($this->options()) && !empty($this->options()) && !in_array($value,$this->options())) {
            $this->error("'{$this->name()}' must be one of the following: ".implode(", ",$this->options()));
            return false;
        }
        return true;
    }

    public function print($printer) {
        if($this->short()) {
            $printer->long($this->short(),$this->name(),$this->description());
        } else {
            $printer->short($this->name(),$this->description());
        }
        return true;
    }

    /**
     * Create a new Param instance
     *
     * - example: *b|book+|The playbook.yml file to execute
     * - example: <>|action|The action to perform
     * - legend: * required, + multiple
     *   
     * @param mixed $syntaxString 
     * @param array $config 
     * @return static 
     * @throws Exception 
     */
    public static function create($syntaxString,$config=[]) {
        $opts = static::parseSyntaxString($syntaxString,$config);
        
        $opts["needsValue"] = true;
        if(strpos($opts["short"],"*") !== false) {
            $opts["required"] = true;
            $opts["short"] = str_replace("*","",$opts["short"]);
        }
        if(strpos($opts["name"],"*") !== false) {
            $opts["required"] = true;
            $opts["name"] = str_replace("*","",$opts["name"]);
        }
        if(strpos($opts["short"],"+") !== false) {
            $opts["multiple"] = true;
            $opts["short"] = str_replace("+","",$opts["short"]);
        }
        if(strpos($opts["name"],"+") !== false) {
            $opts["multiple"] = true;
            $opts["name"] = str_replace("+","",$opts["name"]);
        }
        if(strpos($opts["name"],"-") !== false) {
            $opts["needsValue"] = false;
            $opts["name"] = str_replace("-","",$opts["name"]);
        }
        $opts = array_merge($opts,$config);
        $opts = static::parseOptsArray($opts);
        return new static($opts);
    }
    public static function parseSyntaxString($syntaxString,$opts=[]) {
        $parts = explode("|",$syntaxString);
        if(count($parts) < 2) {
            throw new \Exception("Invalid syntax string, must have at least 2 parts separated by |");
        }
        if(count($parts)==2) {
            $short = "";
            $name = $parts[0];
            $description = $parts[1];
        } else {
            $short = $parts[0];
            $name = $parts[1];
            $description = isset($parts[2]) ? $parts[2] : "";
        }
        
        $opts["short"] = $short;
        $opts["name"] = $name;
        $opts["description"] = $description;
        return $opts;
    }
    public static function parseOptsArray($opts=[]) {
        return $opts;
    }

    public function shortDisplay() {
        if(!$this->short()) {
            return "";
        }
        $short = "-".$this->short();
        if($this->needsValue()) {
            $short .= "<".$this->name().">";
        }
        return $short;
    }
    public function longDisplay() {
        $long = "--".$this->name();
        if($this->needsValue()) {
            $long .= "=<".$this->name().">";
        }
        return $long;
    }

    public function syntax() {
        $valueString = "";
        if($this->needsValue()) {
            $valueString = "<".$this->name().">";
        }
        if($this->short() !== null) {
            return "-".$this->short().' '.$valueString;
        }
        return "-".$this->name().'='.$valueString;
    }
    protected function debug(...$args) {
        CLI::debugCLIformatted("Param",$this->name(),...$args);
    }
}