<?php
namespace boru\cli\printer;

use Exception;

class Row {
    /** @var Cell[] */
    private $cells = [];

    /** @var Table */
    private $table;
    private $padding = 0;
    private $pad = " ";
    private $separator = "";

    private $cellProperties = [];

    public function __construct(...$cells) {
        if(count($cells) === 1 && is_array($cells[0])) {
            $cells = $cells[0];
        }
        if(!empty($cells)) {
            foreach($cells as $index=>$cell) {
                $this->add($cell,$index);
            }
        }
    }
    public function replace($search,$replace) {
        foreach($this->cells as $cell) {
            $cell->replace($search,$replace);
        }
        return $this;
    }
    public function displayLength() {
        $display = $this->display();
        return strlen($display);
    }
    public function display() {
        $this->applyProperties();
        $separator = $this->separator();
        $pad = "";
        if($this->padding() > 0) {
            $pad = str_repeat($this->pad(),$this->padding());
        }
        
        $line = [];
        $lastIndex = max(array_keys($this->cells));
        for($i=0;$i<=$lastIndex;$i++) {
            $line[] = $this->displayCell($i,$pad);
        }
        
        return implode($separator,$line);
    }
    private function displayCell($index,$pad) {
        $cell = $this->cell($index);
        if($cell === null) {
            $width = $this->getWidthFromTable($index);
            return $pad.str_repeat($pad,$width).$pad;
        } else {
            return $pad.$cell->display().$pad;
        }
    }


    public function properties($index,$properties) {
        $this->cellProperties[$index] = $properties;
        return $this;
    }
    public function applyProperties() {
        if(!is_array($this->cellProperties) || empty($this->cellProperties)) {
            return;
        }
        foreach($this->cellProperties as $index=>$properties) {
            foreach($properties as $property=>$value) {
                if(!isset($this->cells[$index])) {
                    continue;
                }
                if(!method_exists($this->cells[$index],$property)) {
                    throw new \Exception("Property '".$property."' does not exist on Cell at index ".$index);
                }
                $this->cells[$index]->$property($value);
            }
        }
    }

    public function getWidthFromTable($index) {
        if(!isset($this->table)) {
            return 0;
        }
        return $this->table->columnWidth($index);
    }

    /**
     * Add a cell to the row
     * @param string|Cell $textOrCell
     * @param int|null $index
     * @return Row
     */
    public function add($textOrCell=null,$index=null) {
        if($index === null) {
            if(empty($this->cells)) {
                $index = 0;
            } else {
                $index = max(array_keys($this->cells))+1;
            }
        }
        if($textOrCell instanceof Cell) {
            $this->cells[$index] = $textOrCell;
        } else {
            $this->cells[$index] = new Cell($textOrCell);
        }
        $this->cells[$index]->index($index);
        $this->cells[$index]->row($this);
        return $this;
    }
    /**
     * A shortcut to return the table object for chaining
     */
    public function endRow() {
        return $this->table();
    }
    /**
     * Get or set the table
     * @param Table|null $table
     * @return Table|Row
     */
    public function table($table=null) {
        if($table === null) {
            return $this->table;
        }
        $this->table = $table;
        return $this;
    }
    /**
     * Get or set a cell
     * @param int $index
     * @param string|null $value
     * @return Cell|null
     */
    public function cell($index,$value=null) {
        if($value !== null) {
            $this->add($value,$index);
        }
        return isset($this->cells[$index]) ? $this->cells[$index] : null;
    }
    /**
     * Get or set the cells
     * @param Cell[]|null $cells
     * @return Cell[]|Row
     */
    public function cells($cells=null) {
        if($cells === null) {
            return $this->cells;
        }
        $this->cells = $cells;
        return $this;
    }

    /**
     * Get or set the padding
     * @param int|null $padding
     * @return int|Row
     */
    public function padding($padding=null) {
        if($padding === null) {
            return $this->padding;
        }
        $this->padding = $padding;
        return $this;
    }
    /**
     * Get or set the padding character
     * @param string|null $pad
     * @return string|Row
     */
    public function pad($pad=null) {
        if($pad === null) {
            return $this->pad;
        }
        $this->pad = $pad;
        return $this;
    }
    /**
     * Get or set the separator
     * @param string|null $separator
     * @return string|Row
     */
    public function separator($separator=null) {
        if($separator === null) {
            return $this->separator;
        }
        $this->separator = $separator;
        return $this;
    }
    /**
     * Get the lengths of the cells
     * @return int[]
     */
    public function lengths() {
        $lengths = [];
        foreach($this->cells as $index=>$cell) {
            $lengths[$index] = $cell->length();
        }
        return $lengths;
    }
    public function setData($data=[]) {
        if(!is_array($data) || empty($data)) {
            return;
        }
        foreach($data as $key=>$value) {
            if(method_exists($this,$key)) {
                $this->$key($value);
            }
        }
    }
    /**
     * Set the width(s) cell(s).. inputs can be an array of [index,width] or a list of widths
     * @param int $index
     * @param int $width
     * @return Row
     */
    public function setWidths(...$args) {
        foreach($args as $k=>$arg) {
            if(is_array($arg)) {
                $this->setOne($arg[0],"width",$arg[1]);
            } else {
                $this->setOne($k,"width",$arg);
            }
        }
        return $this;
    }

    /**
     * Set a property on a cell
     * @param int $index
     * @param string $property
     * @param mixed $value
     * @return Row
     * @throws Exception
     */
    public function setOne($index,$property,$value) {
        if(!isset($this->cells[$index])) {
            return $this;
        }
        if(!method_exists($this->cells[$index],$property)) {
            throw new \Exception("Property '".$property."' does not exist on Cell");
        }
        call_user_func_array([$this->cells[$index],$property],[$value]);
        return $this;
    }
    /**
     * Set a properties on a cell, or cells. Each iinput is an array [index,property,value]
     * @param mixed $index 
     * @param mixed $property 
     * @param mixed $value 
     * @return $this 
     * @throws Exception 
     */
    public function set(...$args) {
        if(count($args) === 3 && !is_array($args[0])) {
            $a = $args[0];
            $b = $args[1];
            $c = $args[2];
            $args = [[$a,$b,$c]];
        }
        foreach($args as $arg) {
            $this->setOne(...$arg);
        }
        return $this;
    }
}