<?php
namespace boru\cli;

use boru\cli\commands\HelpCommand;
use boru\cli\models\Args;
use boru\cli\models\ClassCommand;
use boru\cli\models\Command;
use boru\cli\models\CommandGroup;
use boru\cli\models\Commands;
use boru\cli\models\Params;
use boru\cli\params\Help;
use boru\cli\params\Positional;
use boru\cli\traits\CreateParams;
use boru\dot\Dot;
use boru\output\Output;

class CLI {
    /**
     * This trait is used to create params
     * ->positional($syntaxString,$opts=[])
     * ->flag($syntaxString,$opts=[])
     * ->option($syntaxString,$opts=[])
     */
    use CreateParams;

    /** @var string */
    private $type = "cli";
    /** @var string */
    private $name = "defaultCLI";
    /** @var string */
    private $description;
    /** @var Args */
    private $args;
    /** @var Params */
    private $params;
    /** @var array */
    private $results;
    /** @var Commands */
    private $commands;

    private $groups;

    /** @var CLI */
    private $cli;
    /** @var CLI */
    private $parent;

    private $helpCommand;

    static private $debugMode = false;
    static private $editor;

    private $callback;
    private $noHelp = false;

    public function __construct($syntaxOrArray="", $params=null, $callback=null,$noHelp=false) {
        $this->noHelp = $noHelp;
        if(is_array($syntaxOrArray)) {
            $this->setData($syntaxOrArray);
        } elseif(is_string($syntaxOrArray)) {
            $this->parseSyntaxString($syntaxOrArray);
        }
        $this->helpCommand(HelpCommand::class);
        $this->commands = new Commands();
        $this->commands->cli($this);
        //if(!$this->noHelp) {
            $this->commands->add(new ClassCommand($this->helpCommand));
        //}
        $this->commands()->callback(function($command) {
            if($command === false) {
                $this->print(null,["error"=>"No command specified or invalid command"]);
            } else {
                $this->print(null,["error"=>"Command completed successfully, but callback was either not set or returned true to continue"]);
            }
        });
        
        if($params !== null && $params !== false) {
            $this->params($params);
        }
        if($callback !== null && $callback !== false) {
            $this->callback($callback);
        }
        //$this->cli(false);
    }

    public function type($type=null) {
        if($type === null) {
            return $this->type;
        }
        $this->type = $type;
        return $this;
    }
    public function name($name=null) {
        if($name === null) {
            return $this->name;
        }
        $this->name = $name;
        return $this;
    }
    public function description($description=null) {
        if($description === null) {
            return $this->description;
        }
        $this->description = $description;
        return $this;
    }
    public function cli($cli=null) {
        if($cli === null) {
            return $this->cli;
        }
        $this->debug(__FILE__.":".__LINE__," Setting CLI to",$cli->name());
        $this->cli = $cli;
        return $this->cli;
    }
    public function parent($parent=null) {
        if($parent === null) {
            return $this->parent;
        }
        $this->parent = $parent;
        return $this->parent;
    }

    public function args($args=null) {
        if($args === null) {
            return $this->args;
        }
        if($args === false) {
            $this->args = new Args();
            return $this->args;
        }
        $this->args = new Args($args);
        return $this->args;
    }

    /**
     * @param Params|array|null $params
     * @return Params
     */
    public function params($params=null) {
        if($params === null) {
            if($this->params === null) {
                $this->params = new Params();
                if(!$this->noHelp) {
                    $this->params->add(Help::create());
                }
                $this->params->cli($this);
            }
            return $this->params;
        }
        if($params === false) {
            $this->params = new Params();
            if(!$this->noHelp) {
                $this->params->add(Help::create());
            }
            $this->params->cli($this);
            return $this->params;
        }
        if(!$this->noHelp) {
            array_unshift($params,Help::create());
        }
        foreach($params as $i=>$param) {
            if($param->type() === "command") {
                $this->commandGroup($param->name()."|".$param->description(),$param->params(),$param->callback());
                unset($params[$i]);
            }
        }
        $this->params = new Params($params);
        $this->params->cli($this);
        return $this->params;
    }

    /**
     * @param Commands|array|null $commands
     * @return Commands
     */
    public function commands($commands=null) {
        
        if($commands === null) {
            return $this->commands;
        }
        if($commands === false) {
            $this->commands = new Commands();
            $this->commands->cli($this);
            if(!$this->noHelp) {
                $this->commands->add(new ClassCommand($this->helpCommand));
            }
            return $this->commands;
        }
        if($commands instanceof Commands) {
            $this->commands = $commands;
        } else {
            $this->commands = new Commands($commands);
        }
        $this->commands->cli($this);
        return $this->commands;
    }

    /**
     * @param array|null $results 
     * @return array
     */
    public function results($results=null) {
        if($this->results === null) {
            $this->results = [];
        }
        if(is_array($results)) {
            $this->results = array_replace_recursive($this->results,$results);
        }
        return $this->results;
    }

    public function callback($callback=null) {
        if($callback === null) {
            return $this->callback;
        }
        $this->callback = $callback;
        $this->commands()->callback($callback);
        return $this;
    }

    public function helpCommand($className=null) {
        if($className === null) {
            return $this->helpCommand;
        }
        if(is_object($className)) {
            $className = get_class($className);
        }
        $this->helpCommand = $className;
        return $this;
    }

    public function get($name=null,$default=null) {
        if($name === null) {
            return $this->results;
        }
        return Dot::get($this->results,$name,$default);
    }
    public function set($name,$value) {
        if(strpos($name,".") !== false) {
            Dot::set($this->results,$name,$value);
        }
        else {
            $this->results[$name] = $value;
        }
    }
    public function exists($column=null) {
        if(is_null($column)) {
            return !empty($this->modelData);
        }
        return !is_null(Dot::get($this->results,$column,null));
    }
    public function remove($column) {
        return Dot::delete($this->results,$column);
    }

    public function parseSyntaxString($syntaxString) {
        $parts = explode("|",$syntaxString);
        if(count($parts) < 2) {
            throw new \Exception("Invalid syntax string, must have at least 2 parts separated by |");
        }
        $opts["type"] = "cli";
        $opts["syntax"] = $syntaxString;
        $opts["name"] = $parts[0];
        $opts["description"] = $parts[1];
        $this->setData($opts);
    }

    public function add($command) {
        if($command instanceof CommandGroup) {
            $this->commands()->add($command);
        } else {
            $this->commands()->add($command);
        }
        return $this;
    }

    public function setData($array) {
        if(isset($array["type"])) {
            $this->type($array["type"]);
        }
        if(isset($array["name"])) {
            $this->name($array["name"]);
        }
        if(isset($array["description"])) {
            $this->description($array["description"]);
        }
        if(isset($array["args"])) {
            $this->args($array["args"]);
        }
        if(isset($array["params"])) {
            $this->params($array["params"]);
        }
        if(isset($array["results"])) {
            $this->results($array["results"]);
        }
    }
    
    /**
     * Parse the arguments
     * @param array $args
     * @return true|array
     */
    public function parse($args=null,$previousResults=null) {
        if(php_sapi_name() !== "cli") {
            $msg = static::class."->parse() must be run from the command line";
            $divider = str_repeat("-",strlen($msg)+2);
            throw new \Exception("\n\n".$divider."\n ".static::class."::parse() must be run from the command line\n".$divider."\n\n");
        }
        $this->debug(__FILE__.":".__LINE__," Parsing CLI");
        if($args !== null) {
            $this->args($args);
        } elseif($this->args() === null) {
            $this->args([]);
        }
        if(!is_null($previousResults)) {
            $this->debug(__FILE__.":".__LINE__," Setting previous results from",$this->cli()->name());
            $this->results($previousResults);
        }
        $this->debug(__FILE__.":".__LINE__," Parsing common params");
        if(($result = $this->parseCommonParams()) !== true) {
            if($result === false) {
                return false;
            }
            $this->debug(__FILE__.":".__LINE__," Error parsing common params");
            $this->printError($result);
            return $result;
        }
        $this->debug(__FILE__.":".__LINE__," Parsing commands");
        if(($result = $this->parseCommands()) !== true) {
            $this->debug(__FILE__.":".__LINE__," Returning result");
            return $result;
        }

    }

    protected function parseCommonParams() {
        if($this->params() === null) {
            return true;
        }
        $this->params()->cli($this);
        if(($result = $this->params()->parse($this->args(),$this->commands())) === true) {
            $this->results($this->params()->results());
        } else {
            return $result;
        }
        return true;
    }

    protected function parseCommands() {
        //unconsumed args
        $args = $this->args()->unused();
        $array = [];
        foreach($args as $arg) {
            $array[] = $arg->string();
        }
        $this->debug(__FILE__.":".__LINE__," Unconsumed args: ".implode(" ",$array));
        //$this->commands()->cli($this);
        $result = $this->commands()->parse($args,$this->results());
        return $result;
    }

    public function syntax($commandString=null,$fromCommandGroup=false) {
        if($this->cli() && $this->args()) {
            $script = $this->cli()->syntax($this->name());
            //$commandString= $commandString.$this->name();
        } elseif($this->args() === null) {
            if($this->cli()) {
                $script = $this->cli()->args()->script();
                if($commandString === null) {
                    $commandString = "";
                }
                $commandString= trim($commandString.$this->name());
            } else {
                $script = $this->name();
            }
        } else {
            $script = $this->args()->script();
        }
        $syntax = "";
        if($script === null) {
            $syntax = $this->name();
        } else {
            $script = basename($script);
        }
        $syntax = $script;
        if(substr($script,-4) === ".php") {
            $syntax = "php ".$script;
        }
        if($this->params() !== null) {
            $paramSyntax = $this->params()->syntax();
            if($paramSyntax !== "") {
                $syntax .= " ".$paramSyntax;
            }
        }
        if($commandString !== null) {
            $syntax .= " ".$commandString;
        } else {
            $syntax .= " <command>";
        }
        
        return $syntax;
    }
    public function printError($error) {
        $this->print(null,["error"=>$error]);
    }
    public function banner($printer) {
        $printer->line("{INDENT}".$this->name());
        $printer->line("{INDENT}{INDENT}".$this->description());
        $printer->line("");
        return $this;
    }
    public function error($error,$printer) {
        $printer->addLine("");
        if(is_array($error)) {
            $printer->line("Errors:");
            $table = $printer->table();
            foreach($error as $e) {
                $table->row("",$e)->setWidths(1);
            }
        } else {
            $printer->line("Error: ".$error);
        }
        $printer->addLine("");
        return $this;
    }
    public function print($printer=null,$options=[]) {
        $defaults = [
            "banner"=>false,
            "error"=>null,
            "syntax"=>true,
            "commands"=>true,
            "params"=>true,
            "printer"=>null
        ];
        $options = array_merge($defaults,$options);
        $output = false;
        if($printer === null) {
            $printer = new Printer();
            $output = true;
        }
        $printer->header(1);
        if($options["banner"]) {
            $this->banner($printer);
        }
        if($options["error"] !== null) {
            $this->error($options["error"],$printer);
        }
        if($options["syntax"]) {
            $printer->line("Syntax: ".$this->syntax());
        }
        if($options["printer"]) {
            $printer->fromPrinter($options["printer"]);
        }
        if($options["commands"]) {
            $printer->separator();
            $this->commands()->print($printer);
            if($options["params"] && $this->params() !== null) {
                $printer->separator();
                $this->params()->print($printer);
            }
        }
        $printer->footer(1);
        if($output) {
            $printer->print();
        }
    }

    public static function relPath($from, $to, $ps = DIRECTORY_SEPARATOR) {
        $arFrom = explode($ps, rtrim($from, $ps));
        $arTo = explode($ps, rtrim($to, $ps));
        while(count($arFrom) && count($arTo) && ($arFrom[0] == $arTo[0])) {
            array_shift($arFrom);
            array_shift($arTo);
        }
        if(empty($arFrom) && count($arTo) === 1) {
            return ".".$ps.$arTo[0];
        }
        return str_pad("", count($arFrom) * 3, '..'.$ps).implode($ps, $arTo);
    }

    public static function create($syntaxOrArray="", $params=null, $callback=null) {
        return new CLI($syntaxOrArray, $params, $callback);
    }

    public static function debugCLIformatted($lineFile,$where,$id,...$args) {
        if(static::debugMode()) {
            $lineFile = str_pad(str_replace(dirname(__FILE__),".",$lineFile),30," ");
            $where = str_pad($where,20," ");
            $id = str_pad($id,16," ");
            array_unshift($args,$id);
            array_unshift($args,$where);
            array_unshift($args,$lineFile);
            static::debugCLI(...$args);
        }
    }
    public static function debugCLI(...$args) {
        if(static::debugMode()) {
            Output::outLine(...$args);
        }
    }
    public static function debugMode($mode=null) {
        if($mode === null) {
            return static::$debugMode;
        }
        static::$debugMode = $mode;
    }
    protected function debug($lineFile,...$args) {
        CLI::debugCLIformatted($lineFile,"CLI",$this->type(),...$args);
    }

    public static function editor($editor=null) {
        if($editor !== null) {
            static::$editor = $editor;
        }
        if(static::$editor === null) {
            if(exec("which vim") !== "") {
                static::$editor = "vim";
            } elseif(exec("which nano") !== "") {
                static::$editor = "nano";
            } elseif(exec("which pico") !== "") {
                static::$editor = "pico";
            } elseif(exec("which vi") !== "") {
                static::$editor = "vi";
            }
        }
        return static::$editor;
    }

    public static function edit($content="",$fileName=null) {
        $isTemp = false;
        $editor = static::editor();
        if($editor === null) {
            throw new \Exception("No editor found");
        }
        if(exec("tty") === "") {
            throw new \Exception("No tty found");
        }
        if($fileName === null) {
            $isTemp = true;
            $fileName = tempnam(sys_get_temp_dir(),"cli");
        }
        if($content !== "") {
            file_put_contents($fileName,$content);
        }
        $cmd = $editor." ".$fileName. " > `tty`";
        exec($cmd);
        $content = file_get_contents($fileName);
        if($isTemp) {
            unlink($fileName);
        }
        return $content;
    }
    public static function editFile($fileName) {
        return static::edit("",$fileName);
    }
    public static function promptInput($prompt="",$echo=true,$hash=false) {
        $BACKSPACE = chr(8);
        $ERASE_TO_EOL = "\033[K";
        $userline = [];
        readline_callback_handler_install($prompt, function() {});
        while(true) {
            $keystroke = stream_get_contents(STDIN, 1);
            if($keystroke === false || ord($keystroke) === 10) {
                break;
            } elseif(ord($keystroke) == 127) {
                if(!empty($userline)) {
                    array_pop($userline);
                    if($echo) {
                        fwrite(STDOUT, $BACKSPACE.$ERASE_TO_EOL);
                    }
                }
            } else {
                $userline[] = $keystroke;
                if($echo) {
                    if($hash) {
                        fwrite(STDOUT, "*");
                    } else {
                        fwrite(STDOUT, $keystroke);
                    }
                }
            }
        }
        readline_callback_handler_remove();
        fwrite(STDOUT, PHP_EOL);
        return implode("",$userline);
    }

    /**
     * Prompt user to choose one value from $options array
     *
     * @param  String  $prompt  Text of prompt to display
     * @param  Array   $options Array of valid options. Must be chars.
     * @return String  The valid char selected by the user
     */
    public static function promptChoice($prompt = "Choose One", $options = ['yes', 'no'],$caseSensative=false,$callback=null) {
        /**
         * Force options values to single chars
         */
        $singleChars = [];
        foreach($options as $option) {
            if(!$caseSensative) {
                $char = strtolower(substr($option,0,1));
            } else {
                $char = substr($option,0,1);
            }
            if(in_array($char,$singleChars)) {
                throw new \Exception("Option $char is already in use");
            }
            $singleChars[] = $char;
        } 

        /**
         * Prompt user to choose an option until they select a valid value
         */
        while (true) {

            /**
             * Create a prompt that lists the options
             */
            $showPrompt = "$prompt [".implode(', ', $singleChars)."]".PHP_EOL;

            /**
             * Read one keystroke from the user
             */
            readline_callback_handler_install($showPrompt.PHP_EOL, function() {});
            $keystroke = stream_get_contents(STDIN, 1);
            if(!$caseSensative) {
                $keystroke = strtolower($keystroke);
            }
            /**
             * Return selected value if valid
             */
            if (in_array($keystroke, $singleChars)) {
                echo PHP_EOL;
                if($callback !== null) {
                    return $callback($keystroke);
                }
                return $keystroke;
            }

            /**
             * No valid choice. Show menu again
             */
            echo PHP_EOL;
        }
    }

}