<?php
namespace boru\cli;

use boru\cli\models\Commands;
use boru\cli\models\Args;
use boru\cli\models\Command;
use boru\cli\models\CommandGroup;

abstract class CLIInterface {
    /** @var callable|false */
    private static $CLI_HELP_CALLBACK = false;
    private static $CLI_OBJECT = null;

    public static function CLI_getCli() {
        if (self::$CLI_OBJECT === null) {
            $def = static::CLI_def();
            if ($def instanceof CLI) {
                static::$CLI_OBJECT = $def;
            } else {
                if (isset($def["errorCallback"])) {
                    static::$CLI_HELP_CALLBACK = $def["errorCallback"];
                    unset($def["errorCallback"]);
                } elseif (isset($def["error"])) {
                    static::$CLI_HELP_CALLBACK = $def["error"];
                    unset($def["error"]);
                } elseif (isset($def["error_callback"])) {
                    static::$CLI_HELP_CALLBACK = $def["error_callback"];
                    unset($def["error_callback"]);
                }
                self::$CLI_OBJECT = new CLI($def);
            }
        }
        return self::$CLI_OBJECT;
    }

    public static function parse($args = null) {
        $cli = static::getCli();
        $commandsArray = static::CLI_generateCommands();
        foreach ($commandsArray as $command) {
            if($command instanceof Command) {
                $cli->add($command);
            }
            //$cli->add($command);
        }
        static::$cli = $cli;
        $cli->parse($args);
    }

    /** @var Command */
    private $command;
    public function __construct($command) {
        $this->command = $command;
    }
    public function get($key, $default = null) {
        return $this->command->get($key, $default);
    }
    public function set($key, $value) {
        $this->command->set($key, $value);
        return $this;
    }

    public function args() {
        return $this->command->args();
    }
    public function params() {
        return $this->command->params();
    }
    public function results() {
        return $this->command->results();
    }

    abstract public static function CLI_def();

    protected static $cli;
    protected static $CLI_callback_map = [];

    public static function getCli() {
        if (static::$cli === null) {
            static::$cli = static::CLI_getCli();
        }
        return static::$cli;
    }

    public static function CLI_successCallback($method, $command) {
        if (method_exists(static::class, $method)) {
            $object = new static($command);
            $result = call_user_func_array([$object, $method], [$command]);
        }
    }

    public static function CLI_errorCallback($command) {
        if (static::$CLI_HELP_CALLBACK) {
            return call_user_func_array(static::$CLI_HELP_CALLBACK, [$command]);
        }
    }

    private static function CLI_generateCommands() {
        $methods = get_class_methods(static::class);
        $defsArray = [];
        $callbackArray = [];

        foreach ($methods as $method) {
            if (substr($method, 0, 4) === 'CLI_') {
                continue;
            }

            // Definition methods
            if (strpos($method, "def_") === 0) {
                $commandName = substr($method, 4);
                $def = static::$method();
                $defsArray[$commandName] = $def;
            }

            // Callback methods
            if (strpos($method, "cmd_") === 0) {
                $commandName = substr($method, 4);
                $callbackArray[$commandName] = $method;
            }
        }

        // Build a nested command tree from definitions and callbacks
        $commandTree = [];

        // Insert definitions first so they can set descriptions/params
        foreach ($defsArray as $fullCommandName => $def) {
            self::CLI_insertDefinition($commandTree, explode("_", $fullCommandName), $def);
        }

        // Insert callbacks
        foreach ($callbackArray as $fullCommandName => $callback) {
            self::CLI_insertCallback($commandTree, explode("_", $fullCommandName), $callback);
        }

        // Now finalize the structure (validate and turn partial definitions into groups or commands)
        self::CLI_finalizeTree($commandTree);

        return static::CLI_makeCommands($commandTree);
    }

    /**
     * Recursively insert a definition into the command tree.
     */
    private static function CLI_insertDefinition(&$tree, $parts, $def) {
        $current = array_shift($parts);
        if (!isset($tree[$current])) {
            $tree[$current] = [];
        }

        if (count($parts) === 0) {
            // We're at the final level for this definition
            foreach ($def as $k => $v) {
                $tree[$current][$k] = $v;
            }
        } else {
            // Drill down further
            if (!isset($tree[$current]['commands'])) {
                $tree[$current]['commands'] = [];
            }
            self::CLI_insertDefinition($tree[$current]['commands'], $parts, $def);
        }
    }

    /**
     * Recursively insert a callback into the command tree.
     */
    private static function CLI_insertCallback(&$tree, $parts, $callback) {
        $current = array_shift($parts);
        if (!isset($tree[$current])) {
            $tree[$current] = [];
        }

        if (count($parts) === 0) {
            // At the final level, set the callback
            $tree[$current]['callback'] = $callback;
        } else {
            // Drill down further
            if (!isset($tree[$current]['commands'])) {
                $tree[$current]['commands'] = [];
            }
            self::CLI_insertCallback($tree[$current]['commands'], $parts, $callback);
        }
    }

    /**
     * Finalize the tree:
     * - If a node has no callback but has subcommands, it's a group.
     * - If a node has a callback and no subcommands, it's a leaf command.
     * - If a node has neither callback nor subcommands, it's incomplete.
     */
    private static function CLI_finalizeTree(&$tree, $prefix = '') {
        foreach ($tree as $name => &$data) {
            $fullName = $prefix ? $prefix . "_" . $name : $name;

            // Ensure params is always set
            if (!isset($data['params'])) {
                $data['params'] = [];
            }

            // If there are subcommands, finalize them as well
            if (isset($data['commands'])) {
                self::CLI_finalizeTree($data['commands'], $fullName);

                // If no callback and we have commands, treat as a group
                // This is fine, just ensure we don't require a callback here.
            } else {
                // No subcommands
                if (!isset($data['callback'])) {
                    // No callback, no subcommands, no good
                    // But it's allowed if it was never meant to be a leaf command.
                    // If a def_ method was provided, it means at least it's known.
                    // If there's a description and no callback or commands,
                    // That means it's incomplete. Throw exception.
                    if (!isset($data['description'])) {
                        // This means there's no definition either, incomplete node
                        static::CLI_exception_def($fullName);
                    }
                    // It's a defined command without a callback or subcommands
                    // This is considered incomplete as well.
                    static::CLI_exception_def($fullName);
                }
            }
        }
    }

    private static function CLI_validateCommandsArray($commandArray, $prefix = '') {
        foreach ($commandArray as $commandName => $commandData) {
            $fullName = ($prefix ? $prefix . "_" : "") . $commandName;
            if (!isset($commandData['description']) && !isset($commandData['commands'])) {
                return static::CLI_exception_def($fullName);
            }
            if (!isset($commandData['callback']) && !isset($commandData['commands'])) {
                return static::CLI_exception_def($fullName);
            }
            if (!isset($commandData['params'])) {
                $commandArray[$commandName]['params'] = [];
            }
            if (isset($commandData['commands'])) {
                static::CLI_validateCommandsArray($commandData['commands'], $fullName);
            }
        }
        return true;
    }

    /**
     * Recursively build Command and CommandGroup objects.
     */
    private static function CLI_makeCommands($commandArray, $prefix = '') {
        static::CLI_validateCommandsArray($commandArray, $prefix);
        $commands = [];

        foreach ($commandArray as $commandName => $commandData) {
            $fullName = ($prefix ? $prefix . "_" : "") . $commandName;

            if (isset($commandData['commands']) && !empty($commandData['commands'])) {
                // It's a group
                $commands[] = static::CLI_makeCommandGroup($commandName, $commandData, $fullName);
            } else {
                // It's a leaf command
                $commands[] = static::CLI_makeCommand($commandName, $commandData);
            }
        }

        return $commands;
    }

    private static function CLI_makeCommand($commandName, $commandData) {
        if (!isset($commandData["params"]) || !is_array($commandData["params"])) {
            $commandData["params"] = [];
        }
        $command = new Command(
            [
                "name" => $commandName,
                "description" => $commandData['description']
            ],
            $commandData['params'],
            function ($command) use ($commandData) {
                static::CLI_successCallback($commandData["callback"], $command);
            },
            [static::class, "CLI_errorCallback"]
        );
        return $command;
    }

    private static function CLI_makeCommandGroup($commandName, $commandData, $fullName) {
        $cli = static::getCli();
        if (!isset($commandData["params"]) || !is_array($commandData["params"])) {
            $commandData["params"] = [];
        }
        $description = isset($commandData['description']) ? $commandData['description'] : '';
        $group = $cli->commandGroup($commandName . "|" . $description, $commandData["params"]);

        // Recursively add subcommands
        if (isset($commandData["commands"])) {
            foreach ($commandData["commands"] as $subCommandName => $subCommandData) {
                static::CLI_addSubCommandsRecursively($group, $subCommandName, $subCommandData, $fullName . "_" . $subCommandName);
            }
        }
        return $group;
    }

    private static function CLI_addSubCommandsRecursively($group, $commandName, $commandData, $fullName) {
        if (isset($commandData['commands']) && !empty($commandData['commands'])) {
            // Create a subgroup
            if (!isset($commandData["params"]) || !is_array($commandData["params"])) {
                $commandData["params"] = [];
            }
            $description = isset($commandData['description']) ? $commandData['description'] : '';
            $subGroup = $group->commandGroup($commandName . "|" . $description, $commandData["params"]);
            foreach ($commandData["commands"] as $subName => $subData) {
                static::CLI_addSubCommandsRecursively($subGroup, $subName, $subData, $fullName . "_" . $subName);
            }
        } else {
            // It's a final command
            if (!isset($commandData["params"]) || !is_array($commandData["params"])) {
                $commandData["params"] = [];
            }
            $description = isset($commandData['description']) ? $commandData['description'] : '';
            $group->command(
                $commandName . "|" . $description,
                $commandData["params"],
                function ($command) use ($commandData) {
                    static::CLI_successCallback($commandData["callback"], $command);
                }
            );
        }
    }

    private static function CLI_exception_def($commandName) {
        throw new \Exception("Error in " . static::class . "::CLI_generateCommands()
-----------------------------------------------------------------------------------------------
    Command $commandName is missing a complete definition.
    If you provided def_$commandName(), ensure it returns 'description' and optionally 'params'.
    If this is meant to be a leaf command, you must also define cmd_$commandName().
    If this is meant to be a command group, ensure it has subcommands (def_".$commandName."_subCommand()).
-----------------------------------------------------------------------------------------------\n\n");
        return null;
    }
}