<?php
namespace boru\cli2\Models;

use boru\cli2\Models\Args;
use boru\cli2\Models\Params;

/**
 * Router-style registry and resolver for CLI2.
 */
class CommandRouter
{
    /** @var RouteNode root node (represents the CLI itself) */
    protected $root;

    public function __construct(RouteNode $root)
    {
        $this->root = $root;
    }

    /**
     * Register a route.
     *
     * @param string|string[] $path   e.g. 'user add' or ['user','add']
     * @param Params|array|null $params
     * @param callable|null $handler
     * @param string $description
     * @return RouteNode
     */
    public function addRoute($path, $params = null, $handler = null, $description = '')
    {
        if (is_string($path)) {
            // split on space for convenience: 'user add'
            $segments = preg_split('/\s+/', trim($path));
        } else {
            $segments = $path;
        }

        $node = $this->root;
        $currentPath = array();
        foreach ($segments as $segment) {
            $currentPath[] = $segment;
            $child = $node->getChild($segment);
            if (!$child) {
                $child = new RouteNode($currentPath, '', null, null, true);
                $node->addChild($child);
            }
            $node = $child;
        }

        // Now $node is the leaf for this route
        $nodeDesc = $description !== '' ? $description : $node->description();
        // Apply description if provided
        if ($description !== '') {
            $node->setDescription($description); // we'll add this method
        }

        if ($params instanceof Params) {
            $node->params($params);
        } elseif (is_array($params)) {
            $node->params(new Params($params));
        }

        $node->handler($handler);
        // If it has a handler, it's a leaf command (even if it has children later)
        return $node;
    }

    public function root()
    {
        return $this->root;
    }
}
