<?php
namespace boru\cli2\Params;

abstract class Param
{
    protected $name;        // long name, e.g. "book"
    protected $short;       // short, e.g. "b"
    protected $description; // string
    protected $required = false;
    protected $multiple = false;
    protected $takesValue = false;

    protected $default = null;
    protected $value   = null; // scalar or array depending on $multiple

    public function __construct($name = null, $short = null, $description = '')
    {
        $this->name        = $name;
        $this->short       = $short;
        $this->description = $description;
    }

    // --- basic accessors / mutators ---

    public function name($name = null)
    {
        if ($name === null) {
            return $this->name;
        }
        $this->name = $name;
        return $this;
    }

    public function short($short = null)
    {
        if ($short === null) {
            return $this->short;
        }
        $this->short = $short;
        return $this;
    }

    public function description($description = null)
    {
        if ($description === null) {
            return $this->description;
        }
        $this->description = $description;
        return $this;
    }

    public function required($required = null)
    {
        if ($required === null) {
            return $this->required;
        }
        $this->required = (bool)$required;
        return $this;
    }

    public function multiple($multiple = null)
    {
        if ($multiple === null) {
            return $this->multiple;
        }
        $this->multiple = (bool)$multiple;
        return $this;
    }

    public function takesValue($takesValue = null)
    {
        if ($takesValue === null) {
            return $this->takesValue;
        }
        $this->takesValue = (bool)$takesValue;
        return $this;
    }

    public function default($default = null)
    {
        if (func_num_args() === 0) {
            return $this->default;
        }
        $this->default = $default;
        return $this;
    }

    public function value($value = null)
    {
        if (func_num_args() === 0) {
            return $this->value;
        }

        if ($this->multiple) {
            if ($this->value === null) {
                $this->value = array();
            }
            if (is_array($value)) {
                foreach ($value as $v) {
                    $this->value[] = $v;
                }
            } else {
                $this->value[] = $value;
            }
        } else {
            $this->value = $value;
        }

        return $this;
    }

    // --- matching helpers ---

    public function matchesToken($token)
    {
        if (substr($token, 0, 2) === '--') {
            return $this->name !== null && substr($token, 2) === $this->name;
        }
        if (substr($token, 0, 1) === '-') {
            return $this->short !== null && substr($token, 1) === $this->short;
        }
        return false;
    }

    /**
     * For flag-like params: mark as present.
     * Default behavior: set boolean true or append true if multiple.
     */
    public function setFlag()
    {
        if ($this->multiple) {
            if ($this->value === null) {
                $this->value = array();
            }
            $this->value[] = true;
        } else {
            $this->value = true;
        }
        return $this;
    }

    /**
     * Produce a canonical syntax string for help.
     * Subclasses can override if needed.
     */
    public function syntax()
    {
        $short = $this->short ?: '';
        $name  = $this->name ?: '';
        $desc  = $this->description ?: '';

        $base = $short . '|' . $name . '|' . $desc;

        if ($this->required) {
            $base = '*' . $base;
        }

        // Append '+' to long name if multiple; if no long name, to short.
        if ($this->multiple) {
            if ($name !== '') {
                $base = str_replace('|' . $name . '|', '|' . $name . '+|', $base);
            } elseif ($short !== '') {
                $base = str_replace($short . '|', $short . '+|', $base);
            }
        }

        return $base;
    }
}
