#!/usr/bin/env php
<?php
require __DIR__ . "/../vendor/autoload.php";

use boru\cli2\CLI;
use boru\cli2\CLIContext;
use boru\cli2\Models\Params as ParamsModel;
use boru\cli2\Params\Flag;
use boru\cli2\Params\Option;
use boru\cli2\Params\Positional;
use boru\cli2\Output\BufferOutput;

// Build CLI
$cli = CLI::create('app|CLI2 smoke test');

// Use buffer output for assertions
$buffer = new BufferOutput();
$cli->output($buffer);

// Global params: -d / --debug
$cli->params(array(
    Flag::create('d|debug|Enable debug'),
));

// user group with group-level -u / --username
$cli->group('user|User commands', function ($user) {

    // Group-level params: apply to all user commands
    $user->params(array(
        Option::create('u|username|Default username'),
    ));

    // user add <name>
    $user->route('add|Add a user', array(
        Positional::create('name|Name of user'),
    ), function (CLIContext $ctx) {
        $ctx->output("User add command invoked");
    });

    // user update <name> -f field -v value
    $user->route('update|Update a user', array(
        Positional::create('name|Name of user'),
        Option::create('f|field|Field to update'),
        Option::create('v|value|New value'),
    ), function (CLIContext $ctx) {
        $ctx->output("User update command invoked");
    });
});

// Deploy command
$cli->route('deploy|Deploy the app', array(
    Flag::create('f|force|Force deployment'),
    Option::create('e|env|Environment name'),
), function (CLIContext $ctx) {
    $ctx->output("Deploy command invoked");
    $ctx->output('Global params: ' . var_export($ctx->globalParams(), true));
    $ctx->output('Route params: ' . var_export($ctx->params(), true));
});

// -----------------------------------------------------------------------------
// Assertion harness
// -----------------------------------------------------------------------------

/**
 * Run a case: parse given args and assert expected substrings appear in output.
 *
 * @param CLI          $cli
 * @param BufferOutput $buffer
 * @param string       $label
 * @param string       $args
 * @param string[]     $expectedLines  substrings that must each appear at least once
 * @param string[]     $notExpected    substrings that must NOT appear
 * @return array [ok => bool, message => string]
 */
function run_case(CLI $cli, BufferOutput $buffer, $label, $args, array $expectedLines = array(), array $notExpected = array())
{
    $buffer->clear();

    // Call parse with explicit args string to avoid using real $argv
    $cli->parse($args);

    $output = implode("\n", $buffer->lines());
    $failures = array();

    foreach ($expectedLines as $exp) {
        if (strpos($output, $exp) === false) {
            $failures[] = "Missing expected substring: '{$exp}'";
        }
    }

    foreach ($notExpected as $no) {
        if (strpos($output, $no) !== false) {
            $failures[] = "Unexpected substring present: '{$no}'";
        }
    }

    if (!empty($failures)) {
        return array(
            'ok'      => false,
            "label"   => $label,
            "args"    => $args,
            'message' => $label . " FAILED",
            "failures"=> implode("\n  - ", $failures),
            "output"  => $output,
        );
    }

    return array(
        'ok'      => true,
        "label"   => $label,
        "args"    => $args,
        'message' => $label . " OK",
    );
}

// Define smoke cases
$cases = array(
    array(
        'label'    => 'root help',
        'args'     => '',
        'expect'   => array(
            'app - CLI2 smoke test',
            'Available commands:',
            'deploy',
            'user',
        ),
        'notExpect' => array(
            'Errors:',
            'command invoked',
        ),
    ),
    array(
        'label'    => 'deploy force',
        'args'     => 'deploy -f --env=prod',
        'expect'   => array(
            'Deploy command invoked',
        ),
        'notExpect' => array(
            'Errors:',
        ),
    ),
    array(
        'label'    => 'user namespace help',
        'args'     => 'user',
        'expect'   => array(
            'user - User commands',
            'Subcommands:',
            'user add',
            'user update',
        ),
        'notExpect' => array(
            'command invoked',
        ),
    ),
    array(
        'label'    => 'user add',
        'args'     => 'user add alice',
        'expect'   => array(
            'User add command invoked',
        ),
        'notExpect' => array(
            'Errors:',
        ),
    ),
    array(
        'label'    => 'user add with globals',
        'args'     => '-d user add --username bob charlie',
        'expect'   => array(
            'User add command invoked',
        ),
        'notExpect' => array(
            'Errors:',
        ),
    ),
    array(
        'label'    => 'user update',
        'args'     => 'user update alice -f email -v new@example.com',
        'expect'   => array(
            'User update command invoked',
        ),
        'notExpect' => array(
            'Errors:',
        ),
    ),
);

$print = false; //!--args
$showActual = false; //--actual
$dieOnFailure = false; //--die
foreach($argv as $arg) {
    if ($arg === '--print') {
        $print = true;
    }
    if ($arg === '--actual') {
        $showActual = true;
    }
    if ($arg === '--die') {
        $dieOnFailure = true;
    }
}


// Run all cases
$allOk = true;
foreach ($cases as $case) {
    $result = run_case(
        $cli,
        $buffer,
        $case['label'],
        $case['args'],
        isset($case['expect']) ? $case['expect'] : array(),
        isset($case['notExpect']) ? $case['notExpect'] : array()
    );

    if($print) {
        $status = $result['ok'] ? "OK   " : "FAIL ";
        echo $status . ": " . $case['label'] . "\n";
        echo "Args: " . $case['args'] . "\n";
        if(!$result['ok']) {
            echo "Failures:\n";
            echo "  - " . str_replace("\n", "\n  - ", $result['failures']) . "\n";
        
            if($showActual) {
                echo "\n\nActual Output:\n";
                echo implode("\n", $buffer->lines()) . "\n";
            }
        }
        echo "----------------------\n";
    }

    if (!$result['ok']) {
        $allOk = false;
        if(!$print) fwrite(STDERR, $result['message'] . "\n");
        if($dieOnFailure) {
            exit(1);
        }
    } else {
        if(!$print) fwrite(STDOUT, $result['message'] . "\n");
    }
}

exit($allOk ? 0 : 1);
